from typing import Dict, List, Any, TYPE_CHECKING

import httpx
from lxml import etree as lxmletree

from travelogues_extraction.dataextractors.namespaces import namespaces
from travelogues_extraction.getrecords.session import RecordRetriever
from travelogues_extraction.dataextractors.abstract import AbstractDataExtractor


class Werktitel(AbstractDataExtractor):

    log: List[Dict[str, Any]]
    parend_titles: Dict[str, str]
    upper_level_join_string = ' .- '
    lower_level_join_string = ' ; '

    session: httpx.AsyncClient

    werktitel_xpaths = (
        lxmletree.XPath(
            './marc:datafield[@tag="240" and @ind1="1" and @ind2="0" and marc:subfield[@code="a"]]',
            namespaces=namespaces),
        lxmletree.XPath(
            './marc:datafield[@tag="240" and @ind1="1" and @ind2="0" and marc:subfield[@code="t"]]',
            namespaces=namespaces),
        lxmletree.XPath(
            './marc:datafield[@tag="130" and @ind1="0" and marc:subfield[@code="a"]]',
            namespaces=namespaces),
        lxmletree.XPath(
            './marc:datafield[@tag="130" and @ind1="0" and marc:subfield[@code="t"]]',
            namespaces=namespaces),
    )

    subfield_xpaths = [
        lxmletree.XPath(f'./marc:subfield[@code="{code}" or @code="f" or @code="0"]', namespaces=namespaces)
        for code in ('a', 't', 'a', 't')
    ]

    parent_ac_xpath = lxmletree.XPath(
        './marc:datafield[@tag="773" and @ind1="0" and @ind2="8"]/marc:subfield[@code="w"]',
        namespaces=namespaces)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.parend_titles = {}
        self.log = []

    @classmethod
    def get_columns_names_I_work_on(cls) -> list:
        return ['Werktitel']

    def get_contents(self, element: lxmletree._Element) -> list:
        contents = []
        werktitel_fields = [xpath(element) for xpath in self.werktitel_xpaths]
        for werktitel_field, subbfield_xpath in zip(werktitel_fields, self.subfield_xpaths):
            if len(werktitel_field) == 0:
                continue
            sub_contents = []
            for e in werktitel_field:
                """
                This is not really a list: Eveytime one item. However, if there would be more items => treat as flat list
                """
                sub_contents += subbfield_xpath(e)

            if len(sub_contents) > 0:
                contents.append(
                    self.lower_level_join_string.join([sub_content.text for sub_content in sub_contents]))

        return contents

    async def write(self, record: 'RecordRetriever.Record'):
        async with httpx.AsyncClient() as self.session:
            contents = self.get_contents(record.lxmlelement)
            if len(contents) != 0:
                self.target_dataframe.at[record.ac_number, 'Werktitel'] = self.upper_level_join_string.join(contents)
                return

            # else: try it with parents!
            ac_parent_elements = self.parent_ac_xpath(record.lxmlelement)

            if len(ac_parent_elements) == 0:
                return # and cry

            ac_parent = ac_parent_elements[0].text.replace('(AT-OBV)', '')
            # check if we have requested this ac number already
            if ac_parent in self.parend_titles:
                self.target_dataframe.at[record.ac_number, 'Werktitel'] = self.parend_titles[ac_parent]
                return # and smile

             # else: get the data

            parent_response = await self.session.get(url='https://obv-at-oenb.alma.exlibrisgroup.com/view/sru/43ACC_ONB', params={
                'startRecord': 1,
                'maximumRecords': 1,
                'query': f'alma.local_control_field_009={ac_parent}',
                'version': '1.2',
                'operation': 'searchRetrieve',
                'recordSchema': 'marcxml',
            }
                                               )
            if parent_response.status_code != '200':
                self.log.append({
                    'status_code': parent_response.status_code,
                    'url': parent_response.url,
                    'message': parent_response.text,
                    'ac_child': record.ac_number,
                    'ac_parent': ac_parent,
                })

            try:
                xml = lxmletree.fromstring(parent_response.content)
            except Exception as exception:
                self.log.append({
                    'ac_child': record.ac_number,
                    'ac_parent': ac_parent,
                    'xml_error': exception.__str__(),
                    'xml': parent_response.text
                })

            records = RecordRetriever.record_xpath(xml)

            if len(records) == 0:
                self.log.append({
                    'issue': 'norecords',
                    'ac_child': record.ac_number,
                    'ac_parent': ac_parent
                })
                return

            contents = self.get_contents(records[0])
            if len(contents) != 0:
                self.target_dataframe.at[record.ac_number, 'Werktitel'] = self.upper_level_join_string.join(contents)
