require('dotenv').config();
const express = require('express');
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const cors = require('cors');
const bodyParser = require('body-parser');
const url = require('url');
const crypto = require('crypto');

const app = express();
const port = process.env.PORT || 3000;

class DetailedStats {
	constructor() {
		this.file = path.join(__dirname, '..', 'detailed-stats.json');
		this.stats = {
			rasterTiles: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			vectorTiles: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			fonts: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			styles: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			sprites: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			other: {
				cached: 0,
				fetched: 0,
				total: 0
			},
			totalRequests: 0,
			lastReset: Date.now()
		};
		this.loadStats();
	}
	
	loadStats() {
		try {
			if (fs.existsSync(this.file)) {
				this.stats = JSON.parse(fs.readFileSync(this.file, 'utf8'));
				console.log('Loaded detailed stats:', this.stats);
			} else {
				this.saveStats();
			}
		} catch (error) {
			console.error('Error loading detailed stats:', error);
		}
	}
	
	saveStats() {
		try {
			fs.writeFileSync(this.file, JSON.stringify(this.stats, null, 2));
		} catch (error) {
			console.error('Error saving detailed stats:', error);
		}
	}
	
	identifyRequestType(url) {
		if (url.includes('.webp') || url.includes('/raster/')) {
			return 'rasterTiles';
		} else if (url.includes('vector.pbf')) {
			return 'vectorTiles';
		} else if (url.includes('/fonts/')) {
			return 'fonts';
		} else if (url.includes('/sprite')) {
			return 'sprites';
		} else if (url.includes('?sdk=js')) {
			return 'styles';
		}
		return 'other';
	}
	
	incrementStats(url, isCached) {
		const type = this.identifyRequestType(url);
		this.stats[type].total++;
		if (isCached) {
			this.stats[type].cached++;
		} else {
			this.stats[type].fetched++;
		}
		this.stats.totalRequests++;
		this.saveStats();
	}
	
	getStats() {
		const now = Date.now();
		const uptimeMs = now - this.stats.lastReset;
		const uptime = {
			days: Math.floor(uptimeMs / (1000 * 60 * 60 * 24)),
			hours: Math.floor((uptimeMs % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
			minutes: Math.floor((uptimeMs % (1000 * 60 * 60)) / (1000 * 60))
		};
		
		return {
			...this.stats,
			uptime: `${uptime.days}d ${uptime.hours}h ${uptime.minutes}m`,
			summary: {
				totalCached: Object.values(this.stats).reduce((acc, val) =>
					acc + (val.cached || 0), 0),
				totalFetched: Object.values(this.stats).reduce((acc, val) =>
					acc + (val.fetched || 0), 0)
			}
		};
	}
	
	resetStats() {
		this.stats = {
			rasterTiles: { cached: 0, fetched: 0, total: 0 },
			vectorTiles: { cached: 0, fetched: 0, total: 0 },
			fonts: { cached: 0, fetched: 0, total: 0 },
			styles: { cached: 0, fetched: 0, total: 0 },
			sprites: { cached: 0, fetched: 0, total: 0 },
			other: { cached: 0, fetched: 0, total: 0 },
			totalRequests: 0,
			lastReset: Date.now()
		};
		this.saveStats();
	}
}

// Initialize the detailed stats tracker
const detailedStats = new DetailedStats();

// Add TileCache class
class Cache {
	constructor() {
		this.cacheDir = path.join(__dirname, '..', 'tile-cache');
		this.statsFile = path.join(this.cacheDir, 'cache-stats.json');
		this.stats = {
			hits: 0,
			misses: 0,
			totalRequests: 0,
			bytesStored: 0,
			filesStored: 0,
			lastReset: Date.now(),
			typeStats: {} // Track different file types
		};
		this.ensureCacheDirectory();
		this.loadStats();
	}
	
	ensureCacheDirectory() {
		if (!fs.existsSync(this.cacheDir)) {
			fs.mkdirSync(this.cacheDir, { recursive: true });
		}
	}
	
	loadStats() {
		try {
			if (fs.existsSync(this.statsFile)) {
				this.stats = JSON.parse(fs.readFileSync(this.statsFile, 'utf8'));
				console.log('Loaded cache stats:', this.stats);
			} else {
				this.saveStats();
			}
		} catch (error) {
			console.error('Error loading cache stats:', error);
		}
	}
	
	saveStats() {
		try {
			fs.writeFileSync(this.statsFile, JSON.stringify(this.stats, null, 2));
		} catch (error) {
			console.error('Error saving cache stats:', error);
		}
	}
	
	generateCacheKey(url) {
		return crypto.createHash('md5').update(url).digest('hex');
	}
	
	getCachePath(key) {
		return path.join(this.cacheDir, key);
	}
	
	updateTypeStats(url) {
		const fileType = url.split('.').pop();
		if (!this.stats.typeStats[fileType]) {
			this.stats.typeStats[fileType] = {
				count: 0,
				bytes: 0
			};
		}
		this.stats.typeStats[fileType].count++;
	}
	
	calculateCacheSize() {
		let totalSize = 0;
		let fileCount = 0;
		
		try {
			const files = fs.readdirSync(this.cacheDir);
			files.forEach(file => {
				if (!file.endsWith('.json')) { // Skip stats and header files
					const filePath = path.join(this.cacheDir, file);
					const stats = fs.statSync(filePath);
					totalSize += stats.size;
					fileCount++;
				}
			});
			
			this.stats.bytesStored = totalSize;
			this.stats.filesStored = fileCount;
			this.saveStats();
			
		} catch (error) {
			console.error('Error calculating cache size:', error);
		}
	}
	
	async get(url) {
		const key = this.generateCacheKey(url);
		const cachePath = this.getCachePath(key);
		this.stats.totalRequests++;
		
		try {
			if (fs.existsSync(cachePath)) {
				const stats = fs.statSync(cachePath);
				const ageInDays = (Date.now() - stats.mtime.getTime()) / (1000 * 60 * 60 * 24);
				
				if (ageInDays < 180) {
					console.log('Cache hit for:', url);
					this.stats.hits++;
					this.saveStats();
					return {
						data: fs.readFileSync(cachePath),
						headers: JSON.parse(fs.readFileSync(`${cachePath}.headers.json`))
					};
				} else {
					console.log('Cache expired for:', url);
					fs.unlinkSync(cachePath);
					fs.unlinkSync(`${cachePath}.headers.json`);
					this.calculateCacheSize();
				}
			}
		} catch (error) {
			console.error('Cache read error:', error);
		}
		
		this.stats.misses++;
		this.saveStats();
		return null;
	}
	
	async set(url, data, headers) {
		const key = this.generateCacheKey(url);
		const cachePath = this.getCachePath(key);
		
		try {
			fs.writeFileSync(cachePath, data);
			fs.writeFileSync(`${cachePath}.headers.json`, JSON.stringify(headers));
			console.log('Cached tile:', url);
			
			this.updateTypeStats(url);
			this.calculateCacheSize();
			this.saveStats();
			
		} catch (error) {
			console.error('Cache write error:', error);
		}
	}
	
	// Update getStats to include formatted type stats
	getStats() {
		const hitRate = this.stats.totalRequests ?
			((this.stats.hits / this.stats.totalRequests) * 100).toFixed(2) : 0;
		
		return {
			hits: this.stats.hits,
			misses: this.stats.misses,
			totalRequests: this.stats.totalRequests,
			hitRate: `${hitRate}%`,
			cacheSizeMB: (this.stats.bytesStored / (1024 * 1024)).toFixed(2),
			filesStored: this.stats.filesStored,
			uptime: this.getUptime(),
			typeStats: this.getTypeStats()
		};
	}
	
	getUptime() {
		const uptimeMs = Date.now() - this.stats.lastReset;
		const days = Math.floor(uptimeMs / (1000 * 60 * 60 * 24));
		const hours = Math.floor((uptimeMs % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
		const minutes = Math.floor((uptimeMs % (1000 * 60 * 60)) / (1000 * 60));
		return `${days}d ${hours}h ${minutes}m`;
	}
	
	resetStats() {
		this.stats = {
			hits: 0,
			misses: 0,
			totalRequests: 0,
			bytesStored: 0,
			filesStored: 0,
			lastReset: Date.now(),
			typeStats: {}
		};
		this.calculateCacheSize();
		this.saveStats();
	}
	
	identifyRequestType(url) {
		if (url.includes('.webp') || url.includes('/raster/')) {
			return 'raster';
		} else if (url.includes('vector.pbf')) {
			return 'vector';
		} else if (url.includes('/fonts/')) {
			return 'font';
		} else if (url.includes('/sprite')) {
			return 'sprite';
		} else if (url.includes('?sdk=js')) {
			return 'style';
		}
		return 'other';
	}
	
	updateTypeStats(url, dataSize) {
		const fileType = this.identifyRequestType(url);
		if (!this.stats.typeStats[fileType]) {
			this.stats.typeStats[fileType] = {
				count: 0,
				bytes: 0
			};
		}
		this.stats.typeStats[fileType].count++;
		if (dataSize) {
			this.stats.typeStats[fileType].bytes += dataSize;
		}
	}
	
	async set(url, data, headers) {
		const key = this.generateCacheKey(url);
		const cachePath = this.getCachePath(key);
		
		try {
			fs.writeFileSync(cachePath, data);
			fs.writeFileSync(`${cachePath}.headers.json`, JSON.stringify(headers));
			console.log('Cached:', url);
			
			// Pass the data size to updateTypeStats
			this.updateTypeStats(url, data.length);
			this.calculateCacheSize();
			this.saveStats();
			
		} catch (error) {
			console.error('Cache write error:', error);
		}
	}
	
	// Optional: Add a method to get type-specific stats
	getTypeStats() {
		const stats = {};
		for (const [type, data] of Object.entries(this.stats.typeStats)) {
			stats[type] = {
				count: data.count,
				bytes: data.bytes,
				bytesFormatted: this.formatBytes(data.bytes)
			};
		}
		return stats;
	}
	
	// Helper method to format bytes
	formatBytes(bytes) {
		if (bytes === 0) return '0 B';
		const k = 1024;
		const sizes = ['B', 'KB', 'MB', 'GB'];
		const i = Math.floor(Math.log(bytes) / Math.log(k));
		return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
	}
}

// Initialize the tile cache
const tileCache = new Cache();


class ServiceRateLimiter {
	constructor() {
		this.file = path.join(__dirname, '..', 'service-rate-limits.json');
		this.limits = this.loadEnvLimits();
		this.data = this.initializeData();
		this.loadData();
	}

	loadEnvLimits() {
		// Default limits if env variables are not set
		const defaultLimit = 10000;

		return {
			rasterTiles: {
				cached: parseInt(process.env.LIMIT_RASTER_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_RASTER_UNCACHED || defaultLimit)
			},
			vectorTiles: {
				cached: parseInt(process.env.LIMIT_VECTOR_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_VECTOR_UNCACHED || defaultLimit)
			},
			fonts: {
				cached: parseInt(process.env.LIMIT_FONTS_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_FONTS_UNCACHED || defaultLimit)
			},
			styles: {
				cached: parseInt(process.env.LIMIT_STYLES_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_STYLES_UNCACHED || defaultLimit)
			},
			sprites: {
				cached: parseInt(process.env.LIMIT_SPRITES_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_SPRITES_UNCACHED || defaultLimit)
			},
			other: {
				cached: parseInt(process.env.LIMIT_OTHER_CACHED || defaultLimit),
				uncached: parseInt(process.env.LIMIT_OTHER_UNCACHED || defaultLimit)
			}
		};
	}

	initializeData() {
		const resetTime = Date.now() + (180 * 24 * 60 * 60 * 1000); // 180 days
		const template = {
			cached: 0,
			uncached: 0
		};

		return {
			rasterTiles: { ...template },
			vectorTiles: { ...template },
			fonts: { ...template },
			styles: { ...template },
			sprites: { ...template },
			other: { ...template },
			resetTime
		};
	}

	loadData() {
		try {
			if (fs.existsSync(this.file)) {
				const loadedData = JSON.parse(fs.readFileSync(this.file, 'utf8'));
				// Verify the data structure is valid
				if (this.validateLoadedData(loadedData)) {
					this.data = loadedData;
					console.log('Loaded service rate limit data:', this.data);
				}
			} else {
				this.saveData();
			}
		} catch (error) {
			console.error('Error loading service rate limit data:', error);
		}
	}

	validateLoadedData(data) {
		const requiredServices = ['rasterTiles', 'vectorTiles', 'fonts', 'styles', 'sprites', 'other'];
		const requiredFields = ['cached', 'uncached'];

		return (
			data.resetTime &&
			requiredServices.every(service =>
				data[service] &&
				requiredFields.every(field =>
					typeof data[service][field] === 'number'
				)
			)
		);
	}

	saveData() {
		try {
			fs.writeFileSync(this.file, JSON.stringify(this.data, null, 2));
		} catch (error) {
			console.error('Error saving service rate limit data:', error);
		}
	}

	identifyServiceType(url) {
		if (url.includes('.webp') || url.includes('/raster/')) {
			return 'rasterTiles';
		} else if (url.includes('vector.pbf')) {
			return 'vectorTiles';
		} else if (url.includes('/fonts/')) {
			return 'fonts';
		} else if (url.includes('/sprite') ) {
			return 'sprites';
		} else if (url.includes('?sdk=js')) {
			return 'styles';
		}
		return 'other';
	}

	checkAndResetIfNeeded() {
		const now = Date.now();
		if (now > this.data.resetTime) {
			this.data = this.initializeData();
			this.saveData();
			return true;
		}
		return false;
	}

	incrementCount(url, isCached) {
		this.checkAndResetIfNeeded();
		const serviceType = this.identifyServiceType(url);
		const countType = isCached ? 'cached' : 'uncached';
		this.data[serviceType][countType]++;
		this.saveData();
	}

	checkLimit(url, isCached) {
		this.checkAndResetIfNeeded();
		const serviceType = this.identifyServiceType(url);
		const countType = isCached ? 'cached' : 'uncached';

		return {
			allowed: this.data[serviceType][countType] < this.limits[serviceType][countType],
			current: this.data[serviceType][countType],
			limit: this.limits[serviceType][countType],
			service: serviceType,
			type: countType
		};
	}

	getStats() {
		const now = Date.now();
		const uptimeMs = now - (this.data.resetTime - (180 * 24 * 60 * 60 * 1000));
		const uptime = {
			days: Math.floor(uptimeMs / (1000 * 60 * 60 * 24)),
			hours: Math.floor((uptimeMs % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
			minutes: Math.floor((uptimeMs % (1000 * 60 * 60)) / (1000 * 60))
		};

		const stats = {
			usage: {},
			limits: this.limits,
			uptime: `${uptime.days}d ${uptime.hours}h ${uptime.minutes}m`,
			resetTime: new Date(this.data.resetTime).toISOString()
		};

		// Calculate usage percentages for each service
		for (const [service, counts] of Object.entries(this.data)) {
			if (service !== 'resetTime') {
				stats.usage[service] = {
					cached: {
						count: counts.cached,
						percentage: ((counts.cached / this.limits[service].cached) * 100).toFixed(2)
					},
					uncached: {
						count: counts.uncached,
						percentage: ((counts.uncached / this.limits[service].uncached) * 100).toFixed(2)
					}
				};
			}
		}

		return stats;
	}

	resetStats() {
		this.data = this.initializeData();
		this.saveData();
	}
}

// Initialize the service rate limiter
const serviceLimiter = new ServiceRateLimiter();

// Enable CORS for all routes
app.use(cors());

app.use(bodyParser.raw({ type: '*/*' }));

// You can now use process.env.MAPBOX_TOKEN in your code
const mapboxToken = process.env.MAPBOX_TOKEN;
if (!mapboxToken) {
	console.error('MAPBOX_TOKEN is not set. Please set this environment variable.');
	process.exit(1);
}

const ignore = (req) => {
	
	return true;

	if (req.url.endsWith('.webp')) {
		return true;
	} else if (req.url.endsWith('.vector.pbf')) {
		return true;
	} else if (req.url.endsWith('.pbf')) {
		return true;
	} else {
		return false;
	}
}

console.log('Server starting up...');

class GlobalRateLimiter {
	constructor(limit, resetIntervalMs) {
		this.limit = limit;
		this.resetIntervalMs = resetIntervalMs;
		this.file = path.join(__dirname, '..', 'rate-limit-data.json');
		this.data = { count: 0, resetTime: Date.now() + resetIntervalMs };
		this.loadData();
	}
	
	loadData() {
		console.log('Loading rate limit data...');
		try {
			if (fs.existsSync(this.file)) {
				const fileContent = fs.readFileSync(this.file, 'utf8');
				const loadedData = JSON.parse(fileContent);
				if (loadedData && typeof loadedData.count === 'number' && typeof loadedData.resetTime === 'number') {
					this.data = loadedData;
				}
			}
		} catch (error) {
			console.error('Error loading rate limit data:', error);
		}
		console.log('Current rate limit data:', this.data);
	}
	
	saveData() {
		//console.log('Saving rate limit data...');
		try {
			fs.writeFileSync(this.file, JSON.stringify(this.data, null, 2));
			console.log('Saved rate limit data:', this.data);
		} catch (error) {
			console.error('Error saving rate limit data:', error);
		}
	}
	
	incrementCount(req) {
        // console.log('Checking request for increment:', req.url);
            this.checkAndResetIfNeeded();
			this.data.count++;
			this.saveData();
			// console.log('Incremented count:', this.data.count);
	}
	
	checkAndResetIfNeeded() {
		const now = Date.now();
		if (now > this.data.resetTime) {
			this.data.count = 0;
			this.data.resetTime = now + this.resetIntervalMs;
			console.log('Reset count and time');
			this.saveData();
		}
	}
	
	checkLimit() {
		this.checkAndResetIfNeeded();
		return this.data.count < this.limit;
	}
}

// Create a global rate limiter: 40,000 requests per 180 days
const montly_limit = process.env.MAPBOX_LIMIT ? process.env.MAPBOX_LIMIT : 50000;
console.log('Mapbox limit:', montly_limit)
const globalLimiter = new GlobalRateLimiter(montly_limit, 180 * 24 * 60 * 60 * 1000);

// Middleware to apply global rate limiting to Mapbox API calls
const applyGlobalRateLimit = (req, res, next) => {
	if (globalLimiter.checkLimit()) {
		globalLimiter.incrementCount(req);
		next();
	} else {
		res.status(429).send('Global rate limit exceeded. Please try again later.');
	}
};


app.get('/api/cache/stats', (req, res) => {
	res.json(tileCache.getStats());
});

app.get('/api/stats/service-limits', (req, res) => {
	res.json(serviceLimiter.getStats());
});

app.get('/api/stats/detailed', (req, res) => {
	res.json(detailedStats.getStats());
});

app.post('/api/cache/reset', (req, res) => {
	tileCache.resetStats();
	res.json({ message: 'Cache stats reset successfully' });
});

app.post('/api/stats/reset', (req, res) => {
	detailedStats.resetStats();
	res.json({ message: 'Stats reset successfully' });
});

// Middleware to log all incoming requests
app.use((req, res, next) => {
	console.log(`Received request: ${req.method} ${req.url}`);
	next();
});

// Apply global rate limiter to Mapbox API routes
app.use('/api/mapbox', applyGlobalRateLimit);

// Modify the proxyMapboxRequest function to use the service limiter
const proxyMapboxRequest = async (req, res, mapboxUrl) => {
	console.log('Proxying request to:', mapboxUrl);
	try {
		const parsedUrl = new URL(mapboxUrl);
		parsedUrl.searchParams.delete('access_token');
		parsedUrl.searchParams.append('access_token', process.env.MAPBOX_TOKEN);
		const realUrl = parsedUrl.toString();

		// Check cache first
		const cachedResponse = await tileCache.get(realUrl);

		// Check service-specific rate limit
		const limitCheck = serviceLimiter.checkLimit(realUrl, !!cachedResponse);

		if (!limitCheck.allowed) {
			return res.status(429).json({
				error: 'Service rate limit exceeded',
				details: limitCheck
			});
		}

		if (cachedResponse) {
			Object.keys(cachedResponse.headers).forEach(key => {
				res.setHeader(key, cachedResponse.headers[key]);
			});
			serviceLimiter.incrementCount(realUrl, true);
			detailedStats.incrementStats(realUrl, true);
			return res.send(cachedResponse.data);
		}

		const response = await axios({
			method: req.method,
			url: realUrl,
			headers: {
				...req.headers,
				host: 'api.mapbox.com',
				origin: 'http://localhost:3010',
				referer: 'http://localhost:3010/'
			},
			responseType: 'arraybuffer',
			validateStatus: function (status) {
				return status >= 200 && status < 500;
			}
		});

		Object.keys(response.headers).forEach(key => {
			res.setHeader(key, response.headers[key]);
		});

		if (response.status === 200) {
			await tileCache.set(realUrl, response.data, response.headers);
		}

		serviceLimiter.incrementCount(realUrl, false);
		detailedStats.incrementStats(realUrl, false);
		res.status(response.status).send(response.data);

		console.log('Successfully proxied request to Mapbox');
	} catch (error) {
		console.error('Error proxying request to Mapbox:', error.message);
		if (error.response) {
			console.error('Response status:', error.response.status);
			console.error('Response data:', error.response.data);
		}
		res.status(error.response?.status || 500).send('Error proxying request to Mapbox');
	}
};


// Modify the test route to include detailed stats
app.get('/test-mapbox-limit', (req, res) => {
    const rateLimitInfo = {
        current: globalLimiter.data.count,
        resetTime: new Date(globalLimiter.data.resetTime).toISOString(),
        detailed: detailedStats.getStats()
    };
    res.json(rateLimitInfo);
});

// Vector tile proxy route
app.get('/api/mapbox/v4/:mapLayers/:z/:x/:y.vector.pbf', (req, res) => {
	const { mapLayers, z, x, y } = req.params;
	console.log(`Proxying vector tile request: ${mapLayers}/${z}/${x}/${y}`);
    const mapboxUrl = `https://api.mapbox.com/v4/${mapLayers}/${z}/${x}/${y}.vector.pbf`;
	proxyMapboxRequest(req, res, mapboxUrl);
});

// Map sessions proxy route
app.use('/api/mapbox/map-sessions', (req, res) => {
	console.log('Received Session Request:', req.method, req.url);
	const mapboxUrl = `https://api.mapbox.com/map-sessions${req.url}`;
	proxyMapboxRequest(req, res, mapboxUrl);
});

// General Mapbox API proxy route
app.use('/api/mapbox', async (req, res) => {
	console.log('Received General Mapbox API request:', req.method, req.url);
	const mapboxUrl = `https://api.mapbox.com${req.url}`;
	proxyMapboxRequest(req, res, mapboxUrl);
});

// Serve static files from the dist directory
app.use(express.static(path.join(__dirname, '..', 'dist')));

// Handle any requests that don't match the above
app.get('*', (req, res) => {
	console.log('Serving index.html for route:', req.url);
	res.sendFile(path.join(__dirname, '..', 'dist', 'index.html'));
});

app.listen(port, () => {
	console.log(`Server running on port ${port}`);
});

// Graceful shutdown
process.on('SIGINT', () => {
	console.log('Shutting down server...');
	globalLimiter.saveData();
	process.exit();
});
