const fs = require('fs').promises;
const path = require('path');
const sharp = require('sharp');

async function generateImageList(directoryPath, outputFile) {
	try {
		// Read all files in the directory
		const files = await fs.readdir(directoryPath);
		
		// Filter for image files (you can add more extensions if needed)
		const imageFiles = files.filter(file =>
			['.jpg', '.jpeg', '.png', '.gif'].includes(path.extname(file).toLowerCase())
		);
		
		// Process each image file
		const imageList = await Promise.all(imageFiles.map(async (file) => {
			const filePath = path.join(directoryPath, file);
			
			try {
				// Get image metadata using sharp
				const metadata = await sharp(filePath).metadata();
				
				return {
					src: file, // or you might want to use a full path or URL here
					width: metadata.width,
					height: metadata.height,
					title: path.parse(file).name // use filename (without extension) as title
				};
			} catch (error) {
				console.error(`Error processing ${file}:`, error);
				return null;
			}
		}));
		
		// Filter out any null entries (from errors)
		const validImageList = imageList.filter(img => img !== null);
		
		// Write the image list to a JSON file
		await fs.writeFile(outputFile, JSON.stringify(validImageList, null, 2));
		
		console.log(`Image list has been written to ${outputFile}`);
	} catch (error) {
		console.error('An error occurred:', error);
	}
}

// Usage
const imagesDirectory = '../app/public/highlights';
const outputJsonFile = './imageList.json';

generateImageList(imagesDirectory, outputJsonFile);
