import {useStore} from "../store.js";
import {useRef, useEffect, useState} from "react";
import styles from "../styles.module.css";
import * as d3 from "d3";
import {arrayRange} from "../lib/array-range.js";
import Slider from 'rc-slider';
import '../assets/rcslider.css';
import '../assets/timebrush.css';

export const TimeLine = () => {
	
	const svgRef = useRef()
	const timeline = useStore(state => state.timeline);
	const dim = useStore(state => state.dim);
	
	const filterTimerange = useStore(state => state.filterTimerange)
	const setFilterTimerange = useStore(state => state.setFilterTimerange)
	
	const chartType = useStore(state => state.chartType)
	const mode = useStore(state => state.mode)
	
	const hoveredBook = useStore(state => state.hoveredBook);
	
	const [data1, setData1] = useState()
	const [band, setBand] = useState(10)
	const [maxYscale, setMaxYscale] = useState(7000)
	
	const timeStart = 1500
	const timeStop = 1900
	
	const colors = useStore(state => state.colors);
	
	const [value, setValue] = useState(20)
	
	const bins = arrayRange(timeStart, timeStop, band)
	
	useEffect(() => {
		setBand(value)
	}, [value]);
	
	useEffect(() => {
	}, [mode, chartType, filterTimerange]);
	
	
	useEffect(() => {
		
		let breakpoint = 250
		
		setBand(dim.right.width > breakpoint ? 20 : 20)
		setMaxYscale(dim.right.width > breakpoint ? 4000 : 7000)
		
	}, [dim])
	
	useEffect(() => {
		
		var maximum = 0;
		let newdata = [];
		
		const countColors = (band) => {
			const yellow = band.filter(d => d.Farbklassifizierung === "yellow").length
			const blue = band.filter(d => d.Farbklassifizierung === "blue").length
			const red = band.filter(d => d.Farbklassifizierung === "red").length
			const white = band.filter(d => d.membership === 2).length
			const grey = band.filter(d => d.membership === 0).length
			return ({yellow: yellow, blue: blue, red: red, white: white, grey: grey})
		}
		
		const countMembership = (band) => {
			const eugeniana = band.filter(d => d.membership == 1).length
			const noneugeniana = band.filter(d => d.membership == 0).length
			const uncertain = band.filter(d => d.membership == 2).length
			return ({eugeniana: eugeniana, noneugeniana: noneugeniana, uncertain: uncertain})
		}
		
		bins.map((b, index) => {
			
			var cnt = 0
			var binData = []
			
			timeline.map((d, i) => {
				
				if (index === 0 && i <= b) {
					cnt = cnt + d.length
					binData = [...binData, ...d]
				} else if (index === bins.length - 1 && i >= b) {
					cnt = cnt + d.length
					binData = [...binData, ...d]
				} else if (d.length > 0 && i >= b && i < b + band) {
					cnt = cnt + d.length
					binData = [...binData, ...d]
				}
			})
			
			let yearTag = `${b} - ${b + band - 1}`
			
			if (index === 0) yearTag = `- ${b + band - 1}`;
			if (index === bins.length - 1) yearTag = `${b} -`;
			
			if (cnt > maximum) maximum = cnt;
			
			newdata.push({
				Years: yearTag,
				Value: cnt,
				yearStart: b,
				yearEnd: b + band - 1,
				yearMid: b + band / 2,
				colors: countColors(binData),
				membership: countMembership(binData)
			})
		})
		
		setMaxYscale(maximum * 1.3)
		setData1(newdata)
		
	}, [band]);
	
	useEffect(() => {
		
		if (!data1) return
		if (!timeline) return
		
		const margin = {
				top: dim.right.height * 0.05,
				right: dim.right.width * 0.1,
				bottom: dim.right.height * 0.29,
				left: dim.right.width * 0.1
			},
			width = dim.right.width - margin.left - margin.right,
			height = dim.right.height - margin.top - margin.bottom;
		
		// clear old d3 svg
		d3.select(svgRef.current).selectAll("*").remove();
		
		// append the svg object to the body of the page
		const svg = d3
			.select(svgRef.current)
			.append("svg")
			.attr("width", width + margin.left + margin.right)
			.attr("height", height + margin.top + margin.bottom)
			.append("g")
			.attr("transform", `translate(${margin.left},${margin.top})`);
		
		const x = d3
			.scaleBand()
			.range([0, width])
			.domain(data1.map((d) => d.Years))
			.padding(0.2);
		
		const scalexAxis = d3.scaleLinear([timeStart, timeStop + band], [0, width])
		
		svg
			.append("g")
			.attr("transform", `translate(0, ${height})`)
			.call(d3.axisBottom(scalexAxis).tickFormat((d3.format("d"))))
			.selectAll("text")
			.attr("transform", "translate(-10,0)rotate(-45)")
			.style("text-anchor", "end");
		
		// Add Y axis
		const y = d3.scaleLinear().domain([0, maxYscale]).range([height, 0]);
		svg.append("g").call(d3.axisLeft(y));
		
		svg
			.selectAll("mybar")
			.data(data1)
			.join("rect")
			.attr("class", "barscolor")
			.attr("x", (d) => x(d.Years))
			.attr("y", (d) => y(mode === "colors" ? d.colors.white : d.membership.uncertain))
			.attr("width", x.bandwidth())
			.attr("height", (d) => height - (y(mode === "colors" ? d.colors.white : d.membership.uncertain)))
			.attr("fill", mode === "colors" ? colors.Farbklassifizierung.uncertain : colors.membership[2])
		;
		
		svg
			.selectAll("mybar")
			.data(data1)
			.join("rect")
			.attr("class", "barscolor")
			.attr("x", (d) => x(d.Years))
			.attr("y", (d) => y(mode === "colors" ? d.colors.white + d.colors.yellow : d.membership.uncertain + d.membership.noneugeniana))
			.attr("width", x.bandwidth())
			.attr("height", (d) => height - y(mode === "colors" ? d.colors.yellow : d.membership.noneugeniana))
			.attr("fill", mode === "colors" ? colors.Farbklassifizierung.yellow : colors.membership[0]);
		
		svg
			.selectAll("mybar")
			.data(data1)
			.join("rect")
			.attr("class", "barscolor")
			.attr("x", (d) => x(d.Years))
			.attr("y", (d) => y(mode === "colors" ? d.colors.white + d.colors.yellow + d.colors.red : d.membership.uncertain + d.membership.noneugeniana + d.membership.eugeniana))
			.attr("width", x.bandwidth())
			.attr("height", (d) => height - (y(mode === "colors" ? d.colors.red : d.membership.eugeniana)))
			.attr("fill", mode === "colors" ? colors.Farbklassifizierung.red : colors.membership[1]);
		
		if (mode === 'colors') {
			svg
				.selectAll("mybar")
				.data(data1)
				.join("rect")
				.attr("class", "barscolor")
				.attr("x", (d) => x(d.Years))
				.attr("y", (d) => y(d.colors.white + d.colors.yellow + d.colors.red + d.colors.blue))
				.attr("width", x.bandwidth())
				.attr("height", (d) => height - (y(d.colors.blue)))
				.attr("fill", colors.Farbklassifizierung.blue);
			
			svg
				.selectAll("mybar")
				.data(data1)
				.join("rect")
				.attr("class", "barscolor")
				.attr("x", (d) => x(d.Years))
				.attr("y", (d) => y(d.colors.white + d.colors.yellow + d.colors.red + d.colors.blue + d.colors.grey))
				.attr("width", x.bandwidth())
				.attr("height", (d) => height - (y(d.colors.grey)))
				.attr("fill", colors.Farbklassifizierung.noneugeniana);
		}
		
		d3.selectAll(".barscolor").attr("opacity", d => {
			return filterTimerange?.x0 <= d.yearMid && filterTimerange?.x1 >= d.yearMid ? 1 : 0.2
		});
		
		const brushmargin = {top: 50, right: 0, bottom: 0, left: 0}
		
		const brush = d3.brushX()
			.extent([[brushmargin.left, brushmargin.top], [width - brushmargin.right, height - brushmargin.bottom]])
			.on("end", brushedend);
		
		// Apply the brush to your SVG
		const brushGroup = svg.append("g")
			.attr("class", "brush")
			.call(brush);
		
		// Customize brush appearance
		brushGroup.select(".selection")
			.attr("fill", "#69b3a2")
			.attr("stroke", "#228B22")
			.attr("fill-opacity", 0.3)
			.attr("stroke-width", 2);
		
		brushGroup.selectAll(".handle")
			.attr("fill", "#228B22")
			.attr("stroke", "#228B22")
			.attr("stroke-width", 1);
		
		const scalex = d3.scaleLinear([timeStart, timeStop + band], [brushmargin.left, width - brushmargin.right])
		
		svg.append("g")
			.call(brush)
			.call(brush.move, [filterTimerange?.x0 ? filterTimerange.x0 : 1600, filterTimerange?.x1 ? filterTimerange.x1 : 1700].map(scalex))
			.call(() => {if (!filterTimerange?.x0) setFilterTimerange({x0: 1600, x1: 1700});});
		
		function brushedend(event) {
			const selection = event.selection;
			
			if (!selection) {
				svg.selectAll(".barscolor").attr("opacity", 1)
				setFilterTimerange({x0: timeStart, x1: timeStop})
			}
			
			if (!event.sourceEvent || !selection) return;
			
			const [x0, x1] = selection.map(scalex.invert);
			d3.select(this).transition().call(brush.move, x1 > x0 ? [x0, x1].map(scalex) : null);
			
			if (filterTimerange?.x0 !== x0 || filterTimerange?.x1 !== x1) {
				setFilterTimerange({x0: x0, x1: x1})
			}
			svg.selectAll(".barscolor").attr("opacity", d => {
				return x0 <= d.yearEnd + (d.yearStart - d.yearEnd) / 2 && d.yearMid <= x1 ? 1 : 0.2
			});
		}
		
	}, [data1, timeline, band, dim, mode]);
	
	useEffect(() => {
		
		if (!svgRef.current || !data1) return;
		
		const svg = d3.select(svgRef.current);
		
		function resetHighlight() {
			svg.selectAll(".barscolor")
				.attr("stroke", "none")
				.attr("stroke-width", 0);
		}
		
		function highlightHoveredBook(book) {
			resetHighlight();
			
			if (!book) return;
			
			const bookYear = book['Zz_year'] && book['Zz_year'] !== "0" ? parseInt(book['Zz_year']) : parseInt(book['Veröffentlichungsdatum']);
			const bookMembership = book.membership;
			let bookColor = book.Farbklassifizierung;
			
			// Adjust bookColor based on membership
			if (bookMembership === 0) bookColor = "grey";
			if (bookMembership === 2) bookColor = "white";
			
			// Find the corresponding data point
			const barData = data1.find(d => d.yearStart <= bookYear && d.yearEnd >= bookYear);
			
			if (!barData) return;
			
			let selector;
			if (mode === "colors") {
				switch (bookColor) {
					case "white":
						selector = `.barscolor[fill="${colors.Farbklassifizierung.uncertain}"]`;
						break;
					case "yellow":
						selector = `.barscolor[fill="${colors.Farbklassifizierung.yellow}"]`;
						break;
					case "red":
						selector = `.barscolor[fill="${colors.Farbklassifizierung.red}"]`;
						break;
					case "blue":
						selector = `.barscolor[fill="${colors.Farbklassifizierung.blue}"]`;
						break;
					case "grey":
						selector = `.barscolor[fill="${colors.Farbklassifizierung.noneugeniana}"]`;
						break;
					default:
						console.log("Unexpected bookColor:", bookColor, bookMembership, book);
						return;
				}
			} else if (mode === "memberships") {
				switch (bookMembership) {
					case 0:
						selector = `.barscolor[fill="${colors.membership[0]}"]`;
						break;
					case 1:
						selector = `.barscolor[fill="${colors.membership[1]}"]`;
						break;
					case 2:
						selector = `.barscolor[fill="${colors.membership[2]}"]`;
						break;
					default:
						console.log("Unexpected bookMembership:", bookMembership);
						return;
				}
			} else {
				console.log("Unexpected mode:", mode);
				return;
			}
			
			// Find the specific rectangle for the book's year and color/membership
			const rect = svg.selectAll(selector)
				.filter(d => d.Years === barData.Years);
			
			if (rect.empty()) {
				console.log("No matching rectangle found for:", {
					bookYear,
					bookColor,
					bookMembership,
					barDataYears: barData.Years
				});
				return;
			}
			
			// Highlight the existing rectangle
			rect.attr("stroke", "purple")
				.attr("stroke-width", 3);
		}
		
		if (hoveredBook) {
			highlightHoveredBook(hoveredBook);
		} else {
			resetHighlight();
		}
		
	}, [hoveredBook, data1, mode, colors]);
	
	return (
		<>
			<div className={styles.overlayContainer}
			     style={{fontSize: '0.8rem', paddingTop: '3rem', paddingLeft: '1em', paddingRight: '3rem'}}
			>
				<div className={styles.overlayBox} style={{width: '100%', flexWrap: 'nowrap', alignItems: 'baseline'}}>
					<div style={{width: 'max-content', padding: '1rem'}}>Intervall</div>
					<Slider
						min={1}
						max={100}
						step={1}
						onChange={(val) => setValue(val)}
						defaultValue={20}
						value={value}
					/>
					<div style={{padding: '10px', whiteSpace: 'nowrap'}} title={'Größe des ZeitIntervalls ändern'}>
						{value} {value === 1 ? 'Jahr' : 'Jahre'}
					</div>
				</div>
			</div>
			<svg className={styles.svgDiv} width={'100%'} height={'100%'} ref={svgRef}>
			</svg>
		</>
	)
}
