import styles from "../styles.module.css";
import {useState, useEffect, useRef} from 'react';
import {useMeasure} from "react-use";
import {FloorPlan} from './FloorPlan';
import {OverviewBookshelf} from './OverviewBookshelf';
import {EmptyBookshelf, PlacementBookshelf, PlacementBookshelfSingle, PlacementBookshelfSingleSection,} from './PlacementBookshelf.jsx';
import {useStore} from "../store.js";

export const OvalSvg = () => {
	const [ref, {width, height}] = useMeasure();
	const textRefs = useRef({});
	
	const [side, setSide] = useState(Math.min(width, height));
	const [hoveredAnnotation, setHoveredAnnotation] = useState(null);
	const [showHintKarl, setShowHintKarl] = useState(false);
	
	const mode = useStore(state => state.mode)
	const stats = useStore(state => state.stats);
	const setCurrentShelf = useStore(state => state.setCurrentShelf);
	const setHoveredSection = useStore(state => state.setHoveredSection);
	const setHoldHoveredSection = useStore(state => state.setHoldHoveredSection);
	const setHoveredBook = useStore(state => state.setHoveredBook);
	const showShelfHint = useStore(state => state.showShelfHint);
	const setShowShelfHint = useStore(state => state.setShowShelfHint);
	const zoomShelf = useStore(state => state.zoomShelf);
	const showHighlights = useStore(state => state.showHighlights);
	const setSelectedHighlight = useStore(state => state.setSelectedHighlight);
	const selectedHighlight = useStore(state => state.selectedHighlight);
	
	
	const shelves = ["BE.1", "BE.2", "BE.3", "BE.4", "BE.5", "BE.6", "BE.7", "BE.8", "BE.9", "BE.10", "BE.11", "BE.12"] //, "BE.2", "BE.3", "BE.4", "BE.5", "BE.6", "BE.7", "BE.8", "BE.9", "BE.10", "BE.11", "BE.12"]
	const shelfInfo = {
		"BE.1": {
			id: "BE.1",
			isOval: true,
			label: "BE 1",
			transform: `rotate(-65.5) translate(0 ${-side / 2 * 0.775}) rotate(-9)`
		},
		"BE.2": {
			id: "BE.2",
			isOval: true,
			label: "BE 2",
			transform: `rotate(-42) translate(0 ${-side / 2 * 0.85}) rotate(-14)`
		},
		"BE.3": {
			id: "BE.3",
			isOval: true,
			label: "BE 3",
			transform: `rotate(-21) translate(0 ${-side / 2 * 0.94}) rotate(-17)`
		},
		"BE.4": {
			id: "BE.4",
			isOval: true,
			label: "BE 4",
			transform: `rotate(21) translate(0 ${-side / 2 * 0.94}) rotate(17)`
		},
		"BE.5": {
			id: "BE.5",
			isOval: true,
			label: "BE 5",
			transform: `rotate(42) translate(0 ${-side / 2 * 0.85}) rotate(14)`
		},
		"BE.6": {
			id: "BE.6",
			isOval: true,
			label: "BE 6",
			transform: `rotate(65.5) translate(0 ${-side / 2 * 0.775}) rotate(9)`
		},
		"BE.7": {
			id: "BE.7",
			isOval: true,
			label: "BE 7",
			transform: `rotate(-65.5) translate(0 ${-side / 2 * -0.775}) rotate(-9) rotate(180)`
		},
		"BE.8": {
			id: "BE.8",
			isOval: true,
			label: "BE 8",
			transform: `rotate(-42) translate(0 ${-side / 2 * -0.85}) rotate(-14) rotate(180)`
		},
		"BE.9": {
			id: "BE.9",
			isOval: true,
			label: "BE 9",
			transform: `rotate(-21) translate(0 ${-side / 2 * -0.94}) rotate(-17) rotate(180)`
		},
		"BE.10": {
			id: "BE.10",
			isOval: true,
			label: "BE 10",
			transform: `rotate(21) translate(0 ${-side / 2 * -0.94}) rotate(17) rotate(180)`
		},
		"BE.11": {
			id: "BE.11",
			isOval: true,
			label: "BE 11",
			transform: `rotate(42) translate(0 ${-side / 2 * -0.85}) rotate(14) rotate(180)`
		},
		"BE.12": {
			id: "BE.12",
			isOval: true,
			label: "BE 12",
			transform: `rotate(65.5) translate(0 ${-side / 2 * -0.775}) rotate(9) rotate(180)`
		}
	}
	
	const annotations = [
		{id: 1, shelf: 3, text: "Kircher - Magnetismus der Musik", angle: -36, distance: 1.1},
		{id: 2, shelf: 1, text: "Luther - Biblia", angle: -75, distance: 1.1},
		{id: 3, shelf: 10, text: "Newton - Principia Mathematica", angle: 214, distance: 1.1},
		{id: 4, shelf: 9, text: "Frezier - Spion & Erdbeere", angle: 145, distance: 1.1},
		{id: 5, shelf: 4, text: "Dante - La Divina Commedia", angle: 34, distance: 1.1},
		{id: 6, shelf: 4, text: "Aldrovandi - Elefantenkopf", angle: 41, distance: 1.1},
		{id: 7, shelf: 4, text: "Gessner - Tier-Enzyklopädie", angle: 48, distance: 1.1},
		{id: 8, shelf: 5, text: "Leibniz - Philosophie", angle: 54, distance: 1.1},
		{id: 9, shelf: 5, text: "Vesalius - Anatomieatlas", angle: 60, distance: 1.1},
		{id: 10, shelf: 6, text: "Boym - Chinas Natur", angle: 73, distance: 1.1},
	];
	
	const getAnnotationPosition = (angle, distance) => {
		const radians = (angle - 90) * Math.PI / 180;
		const centerX = width / 2;
		const centerY = height / 2;
		const radiusX = side * 0.5;
		const radiusY = side * 0.6;
		return {
			x: centerX + radiusX * distance * Math.cos(radians),
			y: centerY + radiusY * distance * Math.sin(radians)
		};
	};
	
	const measureText = (id) => {
		const textElement = textRefs.current[id];
		if (textElement) {
			const bbox = textElement.getBBox();
			return { width: bbox.width, height: bbox.height };
		}
		return { width: 0, height: 0 };
	};
	
	useEffect(() => {
		// console.log('Hover state changed:', showHintKarl)
	}, [showHintKarl]);
	
	useEffect(() => {
		let ratio = height / width;
		var offset = 0;
		if (showHighlights) {
			offset = ratio < 0.8 ? -height * 0.15 : -width * ratio * 0.35;
		}
		else {
			offset = ratio < 1.2 ? -height * 0.13 : -width * 0.05;
		}
		setSide(Math.min(width, height) + offset ? Math.min(width, height) + offset : 500) // use 500 as dummy offset
	}, [width, height]);
	
	const handleEmptyClick = () => {
		setCurrentShelf(false)
		setShowShelfHint(false)
		setSelectedHighlight(null)
	}
	
	const handleAnnotationClick = (annotation) => {
		if (selectedHighlight === annotation.id) {
			setCurrentShelf(false)
			setShowShelfHint(false)
			setSelectedHighlight(null)
		}
		else {
			setSelectedHighlight(annotation.id);
		}
	};
	
	
	return (
		<svg
			className={styles.svgDiv}
			width={showHighlights ? '120%' : '100%'} // Make it wider when showHighlights is true
			height={showHighlights ? '105%' : '100%'}
			ref={ref}
		>
			<defs>
				<filter id="drop-shadow" x="-20%" y="-20%" width="140%" height="140%">
					<feGaussianBlur in="SourceAlpha" stdDeviation="2"/>
					<feOffset dx="2" dy="2" result="offsetblur"/>
					<feComponentTransfer>
						<feFuncA type="linear" slope="0.2"/>
					</feComponentTransfer>
					<feMerge>
						<feMergeNode/>
						<feMergeNode in="SourceGraphic"/>
					</feMerge>
				</filter>
			</defs>
			
			<FloorPlan width={width} height={height} side={side}/>
			<g
				onClick={handleEmptyClick}
				onMouseEnter={() => {
					setHoveredSection(false);
					setHoldHoveredSection(false);
					setHoveredBook(false);
				}}
			>
				<rect x={0} y={0} width={'100%'} height={'100%'} fill={'#FFFFFF00'}/>
				
				{/* hint for karl VI*/}
				<rect
					x={width / 2 - side / 2 * 0.2}
					y={height / 2 - side / 2 * 0.12}
					width={side / 2 * 0.4}
					height={side / 2 * 0.24}
					fill="transparent"
					style={{cursor: 'pointer'}}
					onMouseEnter={() => {
						setShowHintKarl(true);
					}}
					onMouseLeave={() => {
						setShowHintKarl(false);
					}}
				/>
				
				{showHintKarl && !showShelfHint &&
				<text
					x={width / 2 - side / 2 * 0.3}
					y={height / 2 - side / 2 * 0.12 - 90}
					dominantBaseline="middle"
					fontSize={side * 0.032}
					fill={'#666'}
				>
					<tspan x={width / 2 - (side / 2) * 0.38} dy="1.2em">
						Ich bin Kaiser Karl VI. und habe die
					</tspan>
					<tspan x={width / 2 - (side / 2) * 0.41} dy="1.2em">
						gesamte Eugeniana-Sammlung 1738
					</tspan>
					<tspan x={width / 2 - (side / 2) * 0.38} dy="1.2em">
						von Prinz Eugens Nichte angekauft.
					</tspan>
				</text>
				}
			
			</g>
			{stats &&
				mode === 'overview' &&
				shelves.map((shelf, i) => {
					if (!(shelf in shelfInfo) || !stats) return null;
					if (shelfInfo[shelf]['isOval']) {
						return (
							<OverviewBookshelf
								key={'overview-shelf' + i}
								width={width}
								height={height}
								side={side}
								shelf={shelfInfo[shelf]}
							/>
						);
					} else {
						return <></>;
					}
				})}
			{stats && (mode === 'colors' || mode === 'memberships') && (
				<>
					<EmptyBookshelf
						key={'placement-shelf'}
						width={width}
						height={height}
						side={side}
						shelves={shelfInfo}
						highlight={selectedHighlight ? annotations[selectedHighlight - 1] : 0}
					/>
					<PlacementBookshelf
						key={'sections-shelf'}
						width={width}
						height={height}
						side={side}
					/>
					{!showHighlights && zoomShelf && (
						<PlacementBookshelfSingle
							key={'sections-shelfSingle'}
							width={width}
							height={height}
							side={side}
						/>
					)}
					{!showHighlights && zoomShelf && (
						<PlacementBookshelfSingleSection
							key={'sections-shelfSingleSection'}
							width={width}
							height={height}
							side={side}
						/>
					)}
				</>
			)}
			{showHighlights &&
				annotations.map((annotation) => {
					const {x, y} = getAnnotationPosition(annotation.angle, annotation.distance);
					const isHovered = hoveredAnnotation === annotation.id;
					const {width: textWidth, height: textHeight} = measureText(annotation.id);
					
					return (
						<g
							key={annotation.id}
							onClick={() => handleAnnotationClick(annotation)}
							onMouseEnter={() => setHoveredAnnotation(annotation.id)}
							onMouseLeave={() => setHoveredAnnotation(null)}
							style={{cursor: 'pointer'}}
						>
							{selectedHighlight === annotation.id && (
								<rect
									x={x > width / 2 ? x - side * 0.01 - side * 0.05 : x - textWidth - side * 0.02}
									y={y - textHeight / 2 - side * 0.01}
									width={textWidth + side * 0.03 + side * 0.05}
									height={textHeight + side * 0.014}
									rx={side * 0.035}
									ry={side * 0.035}
									fill="white"
									stroke='var(--highlightSelected)'
									strokeWidth={isHovered ? 1.5 : 0.5}
									filter="url(#drop-shadow)"
								/>
							)}
							<text
								ref={el => textRefs.current[annotation.id] = el}
								x={x}
								y={y}
								textAnchor={x > width / 2 ? 'start' : 'end'}
								dominantBaseline="middle"
								fontSize={side * (isHovered ? 0.031 : 0.031)}
								fontWeight={isHovered || selectedHighlight === annotation.id ? 'bold' : 'normal'}
								fontFamily="'Saira Semi Condensed', sans-serif"
								fill={isHovered || selectedHighlight === annotation.id ? 'var(--highlightSelected)' : '#666'}
							>
								{annotation.text}
							</text>
							<text
								x={x + (x > width / 2 ? -side * 0.053 : +side * 0.053)}
								y={y}
								textAnchor={x > width / 2 ? 'start' : 'end'}
								dominantBaseline="middle"
								fontSize={side * (isHovered ? 0.052 : 0.052)}
								fontFamily="'WalterTurncoat', sans-serif"
								fill={isHovered || selectedHighlight === annotation.id ? 'var(--highlightSelected)' : '#666'}
							>
								★
							</text>
						</g>
					);
				})}
		</svg>
	);
};
