import styles from "../styles.module.css";
import React, {useState, useEffect, useRef} from 'react';
import {useStore} from "../store";
import {useSpring, animated} from 'react-spring';
import {RiInformation2Line, RiInformationOffLine} from "react-icons/ri";
import {PiNavigationArrowBold} from "react-icons/pi";

export const Onboarding = () => {

	const chartType = useStore(state => state.chartType);
	const dim = useStore(state => state.dim);
	const remInPx = useStore(state => state.remInPx);
	const showHighlights = useStore(state => state.showHighlights);
	const setSelectedHighlight = useStore(state => state.setSelectedHighlight);
	const setShowHighlights = useStore(state => state.setShowHighlights);
	const showStories = useStore(state => state.showStories);
	const showImpressum = useStore(state => state.showImpressum);
	const showTutorial = useStore(state => state.showTutorial);
	const showTour = useStore(state => state.showTour);
	const autoShowTips = useStore(state => state.autoShowTips);

	const [manuallyShown, setManuallyShown] = useState(false);
	const [visitedMember, setVisitedMember] = useState(false);
	const [visitedColor, setVisitedColor] = useState(false);
	const [visitedTime, setVisitedTime] = useState(false);
	const [visitedGeo, setVistedGeo] = useState(false);
	const [visitedLanguage, setVisitedLanguage] = useState(false);
	const [visitedTaxonomy, setVistedTaxonomy] = useState(false);

	const hint = {
		top: dim.left.height / 2 * 0.6,
		left: dim.screen.width / 2 * 0.43,
	};

	const handleCloseStory = () => {
		if (chartType === 'taxonomyCircle') setVistedTaxonomy(true);
		if (chartType === 'geomap') setVistedGeo(true);
		if (chartType === 'time') setVisitedTime(true);
		if (chartType === 'language') setVisitedLanguage(true);
		if (chartType === 'member') setVisitedMember(true);
		if (chartType === 'color') setVisitedColor(true);
		setManuallyShown(false);
	};

	const handleClick = () => {
		setShowHighlights(true);
		setSelectedHighlight(8);
	};
	
	const shouldShowHint = () => {
		if (!autoShowTips) return manuallyShown;
		
		switch (chartType) {
			case 'taxonomyCircle':
				return !visitedTaxonomy;
			case 'geomap':
				return !visitedGeo;
			case 'time':
				return !visitedTime;
			case 'language':
				return !visitedLanguage;
			case 'member':
				return !visitedMember;
			case 'color':
				return !visitedColor;
			default:
				return false;
		}
	};
	
	const handleState = (e) => {
		if (!autoShowTips) {
			setManuallyShown(!manuallyShown);
			return;
		}
		if (chartType === 'taxonomyCircle') setVistedTaxonomy(!visitedTaxonomy);
		if (chartType === 'geomap') setVistedGeo(!visitedGeo);
		if (chartType === 'time') setVisitedTime(!visitedTime);
		if (chartType === 'language') setVisitedTime(!visitedLanguage);
		if (chartType === 'member') setVisitedMember(!visitedMember);
		if (chartType === 'color') setVisitedColor(!visitedColor);
	};
	
	useEffect(() => {
		if (!autoShowTips && manuallyShown) {
				setManuallyShown(false)
		}
	}, [chartType,]);
	
	
	return (
		<>
			<div className={styles.onboardingContainer}
			     style={{
				     top: hint.top,
				     left: hint.left,
				     visibility: (showHighlights || showStories || showTutorial || showImpressum || showTour) ? 'hidden' : 'visible',
			     }}
			>
				{chartType === 'taxonomyCircle' ?
					<Infobutton onClick={handleState} state={!visitedTaxonomy ? 'active' : 'visited'}/>
					: chartType === 'time' ?
						<Infobutton onClick={handleState} state={!visitedTime ? 'active' : 'visited'}/>
						: chartType === 'geomap' ?
							<Infobutton onClick={handleState} state={!visitedGeo ? 'active' : 'visited'}/>
							: chartType === 'language' ?
							<Infobutton onClick={handleState} state={!visitedLanguage ? 'active' : 'visited'}/>
							: chartType === 'member' ?
								<Infobutton onClick={handleState} state={!visitedMember ? 'active' : 'visited'}/>
								: chartType === 'color' ?
									<Infobutton onClick={handleState} state={!visitedColor ? 'active' : 'visited'}/>
									: <Infobutton onClick={handleState} state={'na'}/>
				}

				{chartType === 'taxonomyCircle' &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Die Bücher der Bibliotheca Eugeniana wurden nach dem Philosophen G.W. Leibniz in 20
							Wissensklassen unterteilt.
							Klicken Sie auf eine Kreis, um deren Bücher zu erkunden.
						</p>
						<p>
							Wenn Sie die Maus über ein Buch auf der linken Seite bewegen, wird die
							Wissensklasse violett umrandet. Wenn Sie ein Buch anklicken, können Sie darin schmökern.
						</p>
						<p>
							Sie wollen die Klassifikation im ursprünglichen Latein studieren?
							Rechts oben können Sie sich die Begriffe übersetzen lassen!
						</p>
					</AnimatedHintBox>
				}

				{chartType === 'time' &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Prinz Eugen sammelte über einen längeren Zeitraum ca. 15.000 Druckschriften. Diese Übersicht
							zeigt Ihnen, wann die Bücher im Mitteloval des Prunksaals gedruckt wurden.
						</p>
						<p>
							Wenn Sie am Zeitfenster ziehen, können Sie die Größe des Zeitraums ändern und den Zeitraum
							verschieben.
						</p>
						<img src="timescale.png" alt="Zeitfenster"
						     style={{
							     display: 'block',
							     padding: '0 1rem 0 1rem',
							     margin: '0 auto',
							     width: remInPx * 18,
						     }}
						/>


						<p>
							Die ältesten Werke gehen bis zu den Anfängen des 16. Jahrhunderts zurück.
							<br/>
							Welche Bücher das waren? Finden Sie es heraus!
						</p>
					</AnimatedHintBox>
				}

				{chartType === 'geomap' &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Die Bücher der Bibliotheca Eugeniana wurden über Kooperationen mit mehreren Buchhändlern in
							Europa (speziell: Paris, Amsterdam, Venedig), aber auch Netzwerke mit anderen Gelehrten
							angeschafft.
						</p>
						<p>
							Bewegen Sie die Weltkarte und klicken Sie auf die Orte, um herauszufinden, welche Bücher aus
							Mexiko
							oder Indien ihren Weg in die Bibliotheca Eugeniana gefunden haben.

						</p>
						<p>
							Klicken Sie oberhalb ↑ auf “Eugeniana”, um nur die Ursprungsorte jener Bücher zu sehen, die
							aus der Bibliotheca Eugeniana stammen.
						</p>
					</AnimatedHintBox>
				}
				{chartType === 'language' &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Als gebürtiger Franzose sammelte Prinz Eugen hauptsächlich in seiner Muttersprache und in
							der damaligen Sprache der Wissenschaft: Latein.
						</p>
						<p>Deutsche Bücher finden sich kaum in seiner Sammlung, man erzählt sich sogar, dass er die
							Sprache kaum beherrschte.</p>
						<p>Klicken Sie oberhalb ↑ auf “Eugeniana”, um nur die Sprache jener Bücher zu sehen, die aus der
							Bibliotheca Eugeniana stammen.
						</p>
					</AnimatedHintBox>
				}
				{(chartType === 'member') &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Nicht alle Bücher im Prunksaal sind Teil der Bibliotheca Eugeniana -
							Forschende haben rekonstruiert, welche Bücher dazu gehören:
						</p>
						<div style={{display: 'flex', flex: 'row nowrap', alignItems: 'center'}}>
							<div>
								<img src="book.jpg" alt="Buch"
								     style={{
									     padding: '0 2.5rem 0 1rem',
									     width: remInPx * 4,
								     }}
								/>
							</div>
							<div>
								<ul>
									<li>
										Ein Buch wird im historisch-handgeschriebenen Katalog der Sammlung erwähnt
									</li>
									<li>
										Vorhandensein eines historischen Einbands mit dem Wappen Prinz Eugens
									</li>
									<li>
										Historischer Einband hat die Farbe rot, gelb oder blau.
									</li>
								</ul>
							</div>
						</div>
						<p>
							Nicht immer war die Zuordnung eindeutig möglich. Rot markierte Bücher gehören eindeutig zur Sammlung Eugeniana,
							bei orangen besteht eine gewisse Unsicherheit und graue sind höchstwahrscheinlich nicht Teil der Sammlung Prinz Eugens.
						</p>
					</AnimatedHintBox>
				}
				{(chartType === 'color') &&
					<AnimatedHintBox show={shouldShowHint()} onClose={handleCloseStory}>
						<p>
							Nicht alle Bücher im Prunksaal sind Teil der Bibliotheca Eugeniana -
							Forschende haben rekonstruiert, welche Bücher dazu gehören:
						</p>
						<div style={{display: 'flex', flex: 'row nowrap', alignItems: 'center'}}>
							<div>
								<img src="book.jpg" alt="Buch"
								     style={{
									     padding: '0 2.5rem 0 1rem',
									     width: remInPx * 4,
								     }}
								/>
							</div>
							<div>
								<ul>
									<li>
										Ein Buch wird im historisch-handgeschriebenen Katalog der Sammlung erwähnt
									</li>
									<li>
										Vorhandensein eines historischen Einbands mit dem Wappen Prinz Eugens
									</li>
									<li>
										Historischer Einband hat die Farbe rot, gelb oder blau.
									</li>
								</ul>
							</div>
						</div>
						<p>Lange galt die Vermutung, dass rot für Werke von "Geschichte und Literatur" gewählt wurde, gelb für
							"Naturwissenschaften", und blau für "Theologie und Recht".</p>
					</AnimatedHintBox>
				}
			</div>
		</>
	)
}


const AnimatedHintBox = ({children, onClose, show}) => {
	
	const [shouldRender, setShouldRender] = useState(show);
	
	const dim = useStore(state => state.dim);
	
	const offsetX = dim.screen.width / 2 * 1.2;
	const offsetY = dim.screen.height / 2 * 1.5;
	
	// Animation configuration
	const [springs, api] = useSpring(() => ({
		from: {
			transform: 'scale(0.8) translate(' + offsetX + 'px, ' + offsetY + 'px)',
			opacity: 0
		},
		to: {
			transform: 'scale(1) translate(0px, 0px)',
			opacity: 1
		},
		config: {
			mass: 1,
			tension: 120,
			friction: 14,
			velocity: 0
		}
	}));

	useEffect(() => {
		if (show) {
			setShouldRender(true);
			api.start({
				to: {
					transform: 'scale(1) translate(0px, 0px)',
					opacity: 1
				}
			});
		} else {
			api.start({
				to: {
					transform: 'scale(0.1) translate(' + offsetX + 'px, ' + offsetY + 'px)', // Move diagonally down and right
					opacity: 0
				},
				config: {
					mass: 1,
					tension: 120,
					friction: 14,
					velocity: 0
				}
			});
		}
	}, [show, api]);

	if (!shouldRender) return null;

	return (
		<animated.div
			className={styles.onboardingBox}
			style={{
				...springs,
				transformOrigin: 'center bottom',
				pointerEvents: show ? 'auto' : 'none',
			}}
		>
			<CloseButton onClick={onClose}/>
			{children}
			<TipsToggle/>
		</animated.div>
	);
};

const TipsToggle = () => {
	const setAutoShowTips = useStore(state => state.setAutoShowTips);
	const autoShowTips = useStore(state => state.autoShowTips);
	
	return (
		<label style={{
			display: 'flex',
			alignItems: 'center',
			gap: '0.5rem',
			fontSize: '1rem',
			padding: '0.5rem'
		}}>
			<input
				type="checkbox"
				checked={!autoShowTips}
				onChange={(e) => setAutoShowTips(!e.target.checked)}
			/>
			<span style={{
				display: 'flex',
				alignItems: 'baseline',
				gap: '0.25rem'
			}}>
        Tipps nur über
        <RiInformation2Line
	        size="1.2rem"
	        style={{ position: 'relative', top: '0.2rem' }}
        />
        statt automatisch anzeigen
      </span>
		</label>
	);
};

const CloseButton = ({onClick, style}) => {
	const handleClick = (e) => {
		e.stopPropagation();
		onClick();
	};

	const defaultStyle = {
		position: 'relative',
		width: 'fit-content',
		top: 0,
		left: '100%',
		margin: '-2rem 0 0 0',
		...style
	};

	return (
		<div
			className={styles.close}
			style={defaultStyle}
			onClick={handleClick}
		>
			&#10006;
		</div>
	);
};


const Arrow = () => {

	const dim = useStore(state => state.dim);

	return (<div className={styles.shelfHintContainer}
	             style={{
		             top: dim.top.height + dim.middle.height / 2 * 1,
		             left: dim.left.width + dim.middle.width / 2 * 1.95, // last computation corrects also padding of Oval
		             transform: 'scale(1.5) rotate(130deg)'
	             }}>
			<PiNavigationArrowBold color={'#666'}/>
		</div>
	)
}

const Infobutton = ({onClick, state}) => {
	const dim = useStore(state => state.dim);

	// Animation configuration
	const [springs, api] = useSpring(() => ({
		from: {transform: 'translateY(0px)'},
		config: {
			mass: 1.5,
			tension: 300,
			friction: 12
		}
	}));

	// Watch for state changes and trigger animation
	useEffect(() => {
		if (state !== "visited") return;
		api.start({
			from: {transform: 'translateY(0px)'},
			to: [
				{transform: 'translateY(-20px)'},
				{transform: 'translateY(0px)'}
			]
		});
	}, [state, api]); // Animate whenever state changes

	// Handle click and trigger animation
	const handleClick = () => {
		onClick(state);
	};

	const size = '1.5rem';
	const color = '#666';

	return (
		<animated.div
			title={'info'}
			style={{
				position: 'fixed',
				bottom: '0.5rem',
				left: dim.screen.width / 2 * 1.15,
				width: 'fit-content',
				pointerEvents: 'auto',
				cursor: 'pointer',
				...springs
			}}
		>
			{!state && (
				<RiInformation2Line size={size} color={color} onClick={handleClick}/>
			)}
			{state === 'active' && (
				<RiInformationOffLine size={size} color={color} onClick={handleClick}/>
			)}
			{state === 'visited' && (
				<RiInformation2Line size={size} color={color} onClick={handleClick}/>
			)}
			{state === 'na' && (
				<RiInformation2Line size={size} color={'#ccc'} onClick={handleClick}/>
			)}
		</animated.div>
	);
};
