import React, {useEffect, useRef, useState} from "react";
import styles from "../styles.module.css";
import mirador from "mirador";
import {useStore} from "../store.js";

const createMiradorConfig = (identifier) => ({
	id: "mirador",
	manifests: {
		"https://iiif.onb.ac.at/view/manifest/mirador3/unknown/": {
			provider: "OENB",
		},
	},
	windows: [
		{
			loadedManifest: `https://iiif.onb.ac.at/presentation/ABO/${identifier}/manifest/`,
			canvasIndex: 2,
			thumbnailNavigationPosition: "far-right",
			allowClose: false,
			defaultView: "book",
		},
	],
	workspace: {
		type: "mosaic",
	},
	workspaceControlPanel: {enabled: false},
	selectedTheme: "light",
	themes: {
		light: {
			palette: {
				type: "light",
				primary: {main: "#bbbbbb"},
				secondary: {main: "#FBBE97"},
			},
		},
	},
});

const addCustomCSS = () => {
	const style = document.createElement("style");
	style.textContent = `
	    .mirador-window-sidebar-info-panel {
	      width: auto !important;
	      max-width: 300px;
	    }
	    .mirador-window-sidebar-info-panel h3,
	    .mirador-window-sidebar-info-panel p {
	      word-break: break-word;
	    }
	    .mirador-osd-annotation {
	      word-break: break-word;
	      max-width: 200px;
	    }
	    .MuiTreeItem-root {
	      text-indent: 1.6em !important;
	    }
	    .MuiTreeItem-content {
	      padding-left: 8px !important;
	    }
	    .MuiListItemText-root {
	      text-indent: 1em !important;
	    }
	  `;
	document.head.appendChild(style);
};

export const CombinedViewer = ({identifier, offset}) => {
	const dim = useStore((state) => state.dim);
	const setShowCloseup = useStore((state) => state.setShowCloseup);
	const miradorRef = useRef(null);
	const [isONBViewer, setIsONBViewer] = useState(false);
	const setActiveComponent = useStore(state => state.setActiveComponent);
	
	const handleClose = () => {
		setShowCloseup(false);
		setActiveComponent(null);
	};
	
	useEffect(() => {
		setActiveComponent('combined-viewer');
	}, [setActiveComponent]);
	
	
	useEffect(() => {
		if (!miradorRef.current) {
			const config = createMiradorConfig(identifier);
			miradorRef.current = mirador.viewer(config);
			addCustomCSS();
		}
	}, [identifier]);
	
	const switchToONBViewer = () => setIsONBViewer(true);
	const switchToMiradorViewer = () => setIsONBViewer(false);
	
	const miradorViewerStyle = {
		width: dim.screen.width - dim.left.width - 20 - offset,
		height: dim.left.height,
		position: 'fixed',
		top: dim.screen.height - dim.left.height,
		left: dim.left.width + offset,
		right: '20px',
		bottom: '1em',
		zIndex: 2,
		boxShadow: '0 14px 14px 2px rgba(0, 0, 0, .42)',
	};
	const OenbViewerStyle = {
		width: dim.screen.width - dim.left.width - 32 - offset,
		height: `${dim.left.height - 14}px`,
		border: '1px solid #bbbbbb',
		borderTop: '2px solid #bbbbbb',
		boxShadow: '0 1px 3px 0 rgba(0, 0, 0, .2), 0 1px 1px 0 rgba(0, 0, 0, .2), 0 2px 1px -1px rgba(0, 0, 0, .2)',
		margin: '0px',
		padding: '4px 2px -10px 2px',
		background: 'white',
	};
	
	return (
		<div style={miradorViewerStyle}>
			<div id="mirador" style={{
				display: isONBViewer ? 'none' : 'block'
			}}/>
			{isONBViewer && (
				<div style={{
					border: '6px solid #d6d6d6',
					background: '#d6d6d6',
					margin: '0px',
					padding: '0px',
					width: 'min-content',
					height: 'min-content'
				}}>
					<iframe
						src={`https://digital.onb.ac.at/OnbViewer/viewer.faces?doc=ABO_%2B${identifier}`}
						style={OenbViewerStyle}
						title="ONB Digital Viewer"
						allowFullScreen
					/>
				</div>
			)}
			<button
				className={styles.Button}
				onClick={isONBViewer ? switchToMiradorViewer : switchToONBViewer}
				style={{
					position: 'absolute',
					top: '-1.3em',
					left: '1em',
					zIndex: 4,
					background: 'white',
					border: '4px solid lightgrey',
					padding: '2px 4px 2px 4px',
				}}
			>
				{isONBViewer ? 'ohne Suche' : 'Suche'}
			</button>
			<div className={styles.closeMirador} onClick={handleClose}>
				&#10006;
			</div>
		</div>
	);
};
