import styles from "../styles.module.css";
import 'mapbox-gl/dist/mapbox-gl.css';
import React, {useEffect, useRef, useCallback, useState, useMemo} from 'react';
import mapboxgl from 'mapbox-gl';
import { useStore } from "../store";
import * as d3 from 'd3';
import _ from 'lodash';

// Development mode flag
const DEV_MODE = import.meta.env.VITE_MODE === "development" ? true : false; // Set this to false for production

console.log("DEV_MODE", DEV_MODE)

// Your actual Mapbox token (only used in development mode)
const MAPBOX_TOKEN = import.meta.env.VITE_APP_MAPBOX_ACCESS_TOKEN;

// Disable Mapbox GL's built-in access token requirement
mapboxgl.config.REQUIRE_ACCESS_TOKEN = DEV_MODE ? true : false;

// Set a dummy token to satisfy Mapbox GL JS
mapboxgl.accessToken = DEV_MODE ? MAPBOX_TOKEN : 'dummy.token';


export const GeoMap = () => {
	const mapContainer = useRef(null);
	const map = useRef(null);

	const dim = useStore(state => state.dim);
	const geo = useStore(state => state.geo);
	const setHoveredPlace = useStore(state => state.setHoveredPlace);
	const hoveredPlace = useStore(state => state.hoveredPlace);
	const setFilterPlace = useStore(state => state.setFilterPlace);
	const filterPlace = useStore(state => state.filterPlace);
	const setSelectedCity = useStore(state => state.setSelectedCity);
	const selectedCity = useStore(state => state.selectedCity);
	const hoveredBook = useStore(state => state.hoveredBook);
	const unknownPlacesData = useStore(state => state.unknownPlacesData);
	const showStories = useStore(state => state.showStories);
	const chartType = useStore(state => state.chartType);
	const selectedLegend = useStore(state => state.selectedLegend);
	const mode = useStore(state => state.mode);
	const remInPx = useStore(state => state.remInPx);

	const [processedCities, setProcessedCities] = useState([]);
	const [isMapInitialized, setIsMapInitialized] = useState(false);
	const [fontsize, setFontsize] = useState(false);

	useEffect(()=>{
		setFontsize(remInPx * 0.7)
	}, [remInPx, dim, fontsize])

	useEffect(()=>{
//		console.log(selectedLegend, mode);
	}, [selectedLegend, mode])


	const getProcessedCities = (geo, mode, selectedLegend, scaleRadius, getRadius) => {
		return geo.map((city, id) => {
			const currentRadius = getRadius(city, mode, selectedLegend);
			return {
				...city,
				currentRadius: currentRadius,
				scaledRadius: Math.pow(scaleRadius(Math.max(currentRadius, 1)), 5) + 5,
			};
		});
	};

	// Helper function to get the correct radius based on mode and selected legend
	const getRadius = (city, mode, selectedLegend) => {
		if (!city) return 0;
		if (!selectedLegend) return city.radius || 0; // Return total radius if no legend selected

		if (mode === 'memberships') {
			switch(selectedLegend) {
				case 1: return city.radius_eugeniana || 0;
				case 2: return city.radius_uncertain || 0;
				case 3: return city.radius_noneugeniana || 0;
				default: return city.radius || 0;
			}
		} else if (mode === 'colors') {
			switch(selectedLegend) {
				case 1: return city.radius_yellow || 0;
				case 2: return city.radius_red || 0;
				case 3: return city.radius_blue || 0;
				case 4: return city.radius_uncertain || 0;
				case 5: return city.radius_noneugeniana || 0;
				default: return city.radius || 0;
			}
		}
		return city.radius || 0;
	};

	// assume min and max possible city radius values
	const minRadius = 1; // replace with actual minimum
	const maxRadius = 50; // replace with actual maximum

	// Move scaleRadius setup here
	const scaleRadius = useMemo(() => {
		return d3.scaleLog().domain([minRadius, maxRadius]);
	}, []);


	const customRequestFunction = (url, resourceType) => {
		// console.log('Intercepted request:', url, resourceType);
		if (url.includes('mapbox.com')) {
			let newUrl = url;

			if (url.includes('.vector.pbf')) {
				const regex = /\/v4\/(.+)\/(\d+)\/(\d+)\/(\d+)\.vector\.pbf/;
				const matches = url.match(regex);
				if (matches) {
					const [, mapLayers, z, x, y] = matches;
					newUrl = `${window.location.origin}/api/mapbox/v4/${mapLayers}/${z}/${x}/${y}.vector.pbf`;
				}
			} else if (url.includes('styles')) {
				newUrl = `${window.location.origin}/api/mapbox${url.split('mapbox.com')[1]}`;
			} else if (url.includes('map-sessions')) {
				newUrl = `${window.location.origin}/api/mapbox/map-sessions${url.split('map-sessions')[1]}`;
			} else {
				newUrl = url.replace('https://api.mapbox.com', `${window.location.origin}/api/mapbox`);
			}

			newUrl = newUrl.replace('access_token=dummy.token', 'access_token=dummy_token');

			// console.log('Transformed URL:', newUrl);
			return { url: newUrl };
		}

		// console.log('URL not transformed:', url);
		return { url: url };
	};

	const resetAllFeatureStates = () => {
		if (map.current && map.current.loaded()) {
			processedCities.forEach((city, index) => {
				map.current.setFeatureState(
					{ source: 'cities-source', id: index },
					{ selected: false, hover: false }
				);
			});
		}
	};

	// Single effect to handle all city processing
	useEffect(() => {
		// console.log('Processing cities:', {mode, selectedLegend, geoLength: geo?.length, isMapInitialized});

		if (!geo?.length) return;

		const effectiveMode = mode || 'memberships';
		// Don't set a default for selectedLegend
		const effectiveLegend = selectedLegend;

		const newProcessedCities = getProcessedCities(
			geo,
			effectiveMode,
			effectiveLegend,
			scaleRadius,
			getRadius
		);

		setProcessedCities(newProcessedCities);

		// Only update the map source if the map is already initialized
		if (map.current && map.current.getSource('cities-source')) {
			const updatedGeoJSON = {
				type: 'FeatureCollection',
				features: newProcessedCities.map((city, index) => ({
					type: 'Feature',
					id: index,
					geometry: { type: 'Point', coordinates: [city.lon, city.lat] },
					properties: {
						title: city.city,
						radius: city.currentRadius,
						scaledRadius: city.scaledRadius,
					},
				})),
			};
			map.current.getSource('cities-source').setData(updatedGeoJSON);
		}


	}, [mode, selectedLegend, geo, scaleRadius, isMapInitialized]);

	// Keep your existing effect for updates
	useEffect(() => {
		// console.log('Updating processed cities', mode, selectedLegend);
		if (!mode || !geo.length) return; // Add guard clause

		const newProcessedCities = getProcessedCities(geo, mode, selectedLegend, scaleRadius, getRadius);
		setProcessedCities(newProcessedCities);

		// Update the map source if it exists
		if (map.current && map.current.loaded() && map.current.getSource('cities-source')) {
			const updatedGeoJSON = {
				type: 'FeatureCollection',
				features: newProcessedCities.map((city, index) => ({
					type: 'Feature',
					id: index,
					geometry: { type: 'Point', coordinates: [city.lon, city.lat] },
					properties: {
						title: city.city,
						radius: city.currentRadius,
						scaledRadius: city.scaledRadius,
					},
				})),
			};

			map.current.getSource('cities-source').setData(updatedGeoJSON);
		}
	}, [mode, selectedLegend, geo, scaleRadius]);


	useEffect(() => {

		if (map.current || !processedCities.length) return;

		// console.log(MAPBOX_TOKEN)

		if (DEV_MODE) {
			// In development mode, make direct calls to Mapbox
			map.current = new mapboxgl.Map({
				container: mapContainer.current,
				//				style: 'mapbox://styles/mapbox/light-v10',
				style: `https://api.mapbox.com/styles/v1/mapbox/light-v10?access_token=${MAPBOX_TOKEN}`,
				center: [10.3718643, 48.2081664],
				zoom: 3
			});
		} else {
			// In production mode, use the proxy setup
			map.current = new mapboxgl.Map({
				container: mapContainer.current,
				style: '/api/mapbox/styles/v1/mapbox/light-v10?sdk=js-3.3.0',
				center: [10.3718643, 48.2081664],
				zoom: 3,
				transformRequest: customRequestFunction
			});
		}

		// Wait until map is loaded
		map.current.on('load', function () {
			map.current.addControl(new mapboxgl.NavigationControl());
			map.current.scrollZoom.enable();

			// Hide all city names
			map.current.setLayoutProperty('settlement-label', 'visibility', 'none');

			// Hide all country names
			map.current.setLayoutProperty('country-label', 'visibility', 'none');

			// delete nearly all layers
			//			const layers = map.current.getStyle().layers;
			//			layers.forEach(layer => {
			//				if (layer.id.includes('')) {
			//					// console.log('Country border layer:', layer.id);
			//					map.current.setLayoutProperty(layer.id, 'visibility', 'none');
			//				}
			//			});
			//
			//			map.current.setLayoutProperty('land', 'visibility', 'visible');
			//			map.current.setLayoutProperty('land-cover', 'visibility', 'visible');
			//			map.current.setLayoutProperty('water', 'visibility', 'visible');
			//			map.current.setLayoutProperty('hillshade', 'visibility', 'visible');

			//console.log(map.current.getStyle())

			const cityGeoJSON = {
				type: 'FeatureCollection',
				features: processedCities.map((city, index) => ({
					type: 'Feature',
					id: index,
					geometry: { type: 'Point', coordinates: [city.lon, city.lat] },
					properties: {
						title: city.city,
						radius: city.currentRadius,
						scaledRadius: city.scaledRadius,
					},
				})),
			};

			map.current.addSource('cities-source', { type: 'geojson', data: cityGeoJSON });

			map.current.addLayer({
				id: 'cities-circle',
				type: 'circle',
				source: 'cities-source',
				filter: ['>', ['get', 'radius'], 0], // Add filter here
				paint: {
					'circle-radius': [
						'interpolate',
						['linear'],
						['zoom'],
						1, ['*', ['get', 'scaledRadius'], 0.2],
						20, ['*', ['get', 'scaledRadius'], 2],
					],
					'circle-color': [
						'case',
						['boolean', ['feature-state', 'selected'], false],
						'#FFA500',
						['<', ['get', 'radius'], 10],
						'rgba(0,0,0,0)',
						'#FF004A',
					],
					'circle-opacity': 0.5,
					'circle-stroke-width': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						3,
						['<', ['get', 'radius'], 10],
						0.8,
						1
					],
					'circle-stroke-color': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						'purple',
						['boolean', ['feature-state', 'selected'], false],
						'purple',
						['<', ['get', 'radius'], 10],
						'rgba(255,0,74,0.3)',
						'rgba(255,0,74,0.5)'
					],
				},
			});

			// Extra layer as a "shield"
			map.current.addLayer({
				id: `cities-shield`,
				type: 'circle',
				source: `cities-source`,
				filter: ['>', ['get', 'radius'], 0], // Add filter here
				paint: {
					// Transparent fill
					'circle-color': 'rgba(0, 0, 0, 0)',
					// Stroke color similar to map background
					'circle-stroke-color': 'rgba(0, 0, 0, 0)',
					// Stroke width needs to be enough to cover the gap
					'circle-stroke-width': 4,
					// circle-radius similar to original, but a bit larger
					'circle-radius': [
						'interpolate',
						['linear'],
						['zoom'],
						1, ['*', ['get', 'scaledRadius'], 0.2],
						20, ['*', ['get', 'scaledRadius'], 2],
					],
				}
			});

			map.current.addLayer({
				id: `cities-label`,
				type: 'symbol',
				source: `cities-source`,
				filter: ['>', ['get', 'radius'], 0], // Add filter here
				layout: {
					'text-field': ['get', 'radius'], // convert the value to a string
					'text-anchor': 'center',
					'text-offset': [0, -1],
					'text-size' : fontsize,
					'text-allow-overlap': true,
				},
				paint: {
					'text-color': '#000000', // The color of the text
					"text-halo-color": "#ffffff",
					"text-halo-width": 2,
					'text-opacity': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						1,
						0
					],
				}
			});

			map.current.addLayer({
				id: `cities-label-selected`,
				type: 'symbol',
				source: `cities-source`,
				filter: ['>', ['get', 'radius'], 0], // Add filter here
				layout: {
					'text-field': ['get', 'title'], // convert the value to a string
					'text-anchor': [
						'case',
						['>', ['get', 'radius'], 400],
						'center', // for smaller circles
						'top',  // for large circles
					],
					'text-font': ['Open Sans Bold', 'Arial Unicode MS Bold'],
					'text-offset': [0, 0],
					'text-size' : fontsize,
					'text-allow-overlap': true,
				},
				paint: {
					'text-color': '#222222', // The color of the text
					"text-halo-color": "#ffffff",
					"text-halo-width": 2,
					'text-opacity': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						1,
						0
					],
				}
			});

			map.current.addLayer({
				id: `cities-title`,
				type: 'symbol',
				source: `cities-source`,
				filter: ['all',
					['>', ['get', 'radius'], 0],
					['!=', ['get', 'title'], 'Paris']
				], // Combine radius filter with existing Paris filter
				layout: {
					'text-field': ['get', 'title'], // convert the value to a string
					'text-anchor': [
						'case',
						['>', ['get', 'radius'], 400],
						'center',
						'top',
					],
					'text-offset': [0, 0],
					'text-size' : fontsize,
					'text-padding': 20,
					'symbol-spacing': 250,
					'text-priority': ['get', 'radius']
					//	'visibility': 'none' // the label is not visible initially
				},
				paint: {
					'text-color': '#000000', // The color of the text
					'text-opacity': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						0,
						1
					],
				}
			});


			// Separate layer exclusively for "Paris" label
			map.current.addLayer({
				id: `paris-label`,
				type: 'symbol',
				source: `cities-source`,
				filter: ['all',
					['>', ['get', 'radius'], 0],
					['==', ['get', 'title'], 'Paris']
				], // Combine radius filter with existing Paris filter
				layout: {
					'text-field': ['get', 'title'],
					'text-anchor': 'center',
					'text-offset': [0, 0],
					'text-size' : fontsize,
				},
				paint: {
					'text-color': '#000000',
					'text-opacity': [
						'case',
						['boolean', ['feature-state', 'hover'], false],
						0,
						1
					],
				}
			});

			//			let selectedCityId = null;
			let cityClicked = false;

			// City-specific click handler
			map.current.on('click', 'cities-circle', function(e) {
				cityClicked = true;

				const cityName = e.features[0].properties.title;
				const clickedCityId = e.features[0].id;

				// Reset all feature states
				resetAllFeatureStates();

				// Toggle selection for the clicked city
				if (clickedCityId === selectedCity) {
					setSelectedCity(null);
					setFilterPlace(null);
				} else {
					setSelectedCity(clickedCityId);
					setFilterPlace(cityName);
					map.current.setFeatureState({ source: 'cities-source', id: clickedCityId }, { selected: true });
				}
			});

			// General map click handler
			map.current.on('click', function (e) {
				if (!cityClicked) {
					// console.log('Clicked on an empty part of the map', selectedCity);
					resetAllFeatureStates();
					setSelectedCity(null);
					setFilterPlace(null);
					setHoveredPlace(null);
				}
				// Reset the flag after handling the click
				cityClicked = false;
			});

			let hoveredId = null;

			// When the user moves over a feature, update hover state
			map.current.on('mousemove', 'cities-circle', function(e) {
				map.current.getCanvas().style.cursor = 'crosshair';  // Changed from 'pointer' to 'crosshair'
				if (useStore.getState().filterPlace) {
					return
				}
				
				if (hoveredId) {
					// Reset hover state for previous feature
					map.current.setFeatureState({ source: 'cities-source', id: hoveredId }, { hover: false });
				}
				hoveredId = e.features[0].id;
				
				if (!useStore.getState().filterPlace) {
					// Set hover state for current feature
					map.current.setFeatureState({source: 'cities-source', id: hoveredId}, {hover: true});
					const cityName = e.features[0].properties.title;
					setHoveredPlace(cityName)
				}
			});
			
			// When the user moves away from a feature, reset hover state
			map.current.on('mouseout', 'cities-circle', function() {
				map.current.getCanvas().style.cursor = 'pointer';  // Reset cursor to default when not hovering
				if (useStore.getState().filterPlace) {
					return
				}
				if (hoveredId) {
					map.current.setFeatureState({ source: 'cities-source', id: hoveredId }, { hover: false });
				}
				hoveredId = null;
				setHoveredPlace(null)
			});

			setIsMapInitialized(true);

		});
	}, [processedCities]);

	const updateHoveredCity = useCallback(
		_.throttle((book) => {

			if (map.current && map.current.loaded() && !hoveredPlace) {

				// Reset all feature states
				resetAllFeatureStates();

				if (!book) return;

				const cityForBook = processedCities.find(city => city.city === book.Orte_corr);
				if (cityForBook) {
					const cityIndex = processedCities.indexOf(cityForBook);
					map.current.setFeatureState(
						{ source: 'cities-source', id: cityIndex },
						{ hover: true }
					);
				}
			}
		}, 100),
		[resetAllFeatureStates, processedCities, setHoveredPlace]
	);

	useEffect(() => {
		updateHoveredCity(hoveredBook);
	}, [hoveredBook, updateHoveredCity]);


	useEffect(() => {
		if (map.current && map.current.loaded()) {
			resetAllFeatureStates();
			if (selectedCity !== null) {
				map.current.setFeatureState({ source: 'cities-source', id: selectedCity }, { selected: true });
			}
		}
	}, [selectedCity]);

	useEffect(() => {
		if (selectedCity && (filterPlace === false || filterPlace === null)) {
			map.current.setFeatureState({ source: 'cities-source', id: selectedCity }, { hover: false });
			map.current.setFeatureState({ source: 'cities-source', id: selectedCity }, { selected: false });
			setSelectedCity(null)
			setHoveredPlace(null)
		}
	}, [filterPlace]);

	// Helper to determine the correct size based on context
	const getMapDimensions = () => {
		// If we're in the story view
		if (showStories) {
			return {
				width: dim.screen.width * 0.5,
				height: dim.screen.height * 0.5,
				margin: dim.screen.width * 0.04,
				marginTop: dim.right.width * 0.04,
			};
		}
		// If we're in the map view mode
		if (chartType === 'geomap') {
			return {
				width: dim.right.width * 0.92,
				height: dim.right.height * 0.84,
				margin: dim.right.width * 0.04,
				marginTop: dim.right.width * 0.10,
			};
		}
	};

	useEffect(() => {
		if (dim && map.current) {
			const timer = setTimeout(() => {
				const container = map.current.getContainer();
				const dimensions = getMapDimensions();
				// Explicitly set dimensions on the container
				if (container) {
					container.style.width = `${dimensions.width}px`;
					container.style.height = `${dimensions.height}px`;
					container.style.margin = `${dimensions.margin}px`;
					container.style.marginTop = `${dimensions.marginTop}px`;
				}

				map.current.resize();
			}, 500);

			return () => clearTimeout(timer);
		}
	}, [dim, geo, showStories, chartType]); // Add chartType to dependencies

	return (
		<div
			className={styles.GeoMap}
			ref={mapContainer}
			style={getMapDimensions()}
		>
			<MapLegend unknownPlacesData={unknownPlacesData} />
		</div>
	);
};

const MapLegend = ({ unknownPlacesData }) => {

	return (
		<div style={{
			position: 'absolute',
			bottom: '1.4em',
			left: '50%',
			transform: 'translateX(-50%)',
			backgroundColor: 'white',
			padding: '8px',
			borderRadius: '4px',
			display: 'flex',
			boxShadow: '0 2px 4px rgba(0,0,0,0.1)',
			zIndex: 1, // Add this line
			alignItems: 'center', // Add this line
			whiteSpace: 'nowrap',
		}}>
			<div style={{ display: 'flex', alignItems: 'center', marginBottom: '0px' }}>
				<div style={{
					width: '8px',
					height: '8px',
					marginRight: '4px',
					borderRadius: '50%',
					border: '1.5px solid rgba(255,0,74,0.3)',
				}}></div>
				<span style={{ fontSize: '12px' }}>Orte &lt; 10</span>
			</div>
			<div style={{ display: 'flex', alignItems: 'center', paddingLeft: '1.2em'}}>
				<div style={{
					width: '12px',
					height: '12px',
					marginRight: '4px',
					borderRadius: '50%',
					backgroundColor: 'rgba(255,0,74,0.5)',
				}}></div>
				<span style={{ fontSize: '12px' }}>Orte ≥ 10</span>
			</div>
			<div style={{ display: 'flex', alignItems: 'center', paddingLeft: '1.2em'}}>
				<span style={{ fontSize: '12px' }}>unbekannter Ort: {unknownPlacesData.count}</span>
			</div>
		</div>
	);
};
