import styles from "../styles.module.css";
import React, { useEffect, useRef, useState } from 'react';
import mapboxgl from 'mapbox-gl';
import 'mapbox-gl/dist/mapbox-gl.css';
import {useStore} from "../store.js";

// Development mode flag
const DEV_MODE = import.meta.env.VITE_MODE === "development" ? true : false; // Set this to false for production

export const Intro = () => {

	const store = useStore();
	
	const mapContainer = useRef(null);
	const map = useRef(null);
	
	const showIntro = useStore(state => state.showIntro)
	
	const [isLoaded, setIsLoaded] = useState(false);
	const [error, setError] = useState(null);
	const [isFlying, setIsFlying] = useState(false);
	const [isFading, setIsFading] = useState(false);
	const [isRemoved, setIsRemoved] = useState(false);
	const [allowSkip, setAllowSkip] = useState(false);

	// console.log("DEV_MODE", DEV_MODE)

	// Your actual Mapbox token (only used in development mode)
	const MAPBOX_TOKEN = import.meta.env.VITE_APP_MAPBOX_ACCESS_TOKEN;

	// Disable Mapbox GL's built-in access token requirement
	mapboxgl.config.REQUIRE_ACCESS_TOKEN = DEV_MODE ? true : false;

	// Set a dummy token to satisfy Mapbox GL JS
	mapboxgl.accessToken = DEV_MODE ? MAPBOX_TOKEN : 'dummy.token';

	const customRequestFunction = (url, resourceType) => {
		// console.log('Intercepted request:', url, resourceType);
		if (url.includes('mapbox.com')) {
			let newUrl = url;

			if (url.includes('.vector.pbf')) {
				const regex = /\/v4\/(.+)\/(\d+)\/(\d+)\/(\d+)\.vector\.pbf/;
				const matches = url.match(regex);
				if (matches) {
					const [, mapLayers, z, x, y] = matches;
					newUrl = `${window.location.origin}/api/mapbox/v4/${mapLayers}/${z}/${x}/${y}.vector.pbf`;
				}
			} else if (url.includes('styles')) {
				newUrl = `${window.location.origin}/api/mapbox${url.split('mapbox.com')[1]}`;
			} else if (url.includes('map-sessions')) {
				newUrl = `${window.location.origin}/api/mapbox/map-sessions${url.split('map-sessions')[1]}`;
			} else {
				newUrl = url.replace('https://api.mapbox.com', `${window.location.origin}/api/mapbox`);
			}

			newUrl = newUrl.replace('access_token=dummy.token', 'access_token=dummy_token');

			// console.log('Transformed URL:', newUrl);
			return { url: newUrl };
		}

		// console.log('URL not transformed:', url);
		return { url: url };
	};
	
	useEffect(() => {
		if (!showIntro) {
			setIsRemoved(true)
		}
	}, [showIntro]);
	

	useEffect(()=>{
		if (store.data !== null) {
			// console.log('allowskip = true')
			setTimeout(()=>{
				setAllowSkip(true)
			}, 3000)
		}
	}, [store.data])
	
	const handleSkip = () => {
		setIsFading(true);
	};
	
	// Handle cleanup after fade out
	useEffect(() => {
		if (isFading) {
			const timer = setTimeout(() => {
				if (map.current) {
					map.current.remove();
					map.current = null;
				}
				setIsRemoved(true);
			}, 1000); // Match this with the fade-out duration
			
			return () => clearTimeout(timer);
		}
	}, [isFading]);
	
	useEffect(() => {
		
		if (map.current) {
			console.log("Map instance already exists");
			return;
		}
		
		const initializeMap = async () => {
			try {

				mapboxgl.accessToken = MAPBOX_TOKEN;
				
				const config = {
					container: mapContainer.current,
					style: DEV_MODE ? `mapbox://styles/mapbox/satellite-v9?access_token=${MAPBOX_TOKEN}` : `mapbox://styles/mapbox/satellite-v9`,
					center: [16.3810, 48.1910], // Upper Belvedere coordinates
					zoom: 17.5,
					pitch: 60, // Tilt for 3D view
					bearing: 172, // Rotate to face the front
					projection: 'globe',
					minZoom: 2,
					maxZoom: 20,
					renderWorldCopies: true,
					transformRequest: DEV_MODE ? ()=>{} : customRequestFunction
				};
				// console.log("Creating new map instance with config:", config);
				
				// Create map instance
				map.current = new mapboxgl.Map(config);
				
				// console.log("Map instance created:", !!map.current);
				
				// Add moveend handler to detect end of flyTo
				map.current.on('moveend', ({ originalEvent }) => {
					if (!originalEvent) {
						//						console.log("FlyTo animation completed!");
						setIsFading(true);
					}
				});
				
				// Add error handler
				map.current.on('error', (e) => {
					// console.error("Mapbox error:", e);
					setError(e.error?.message || "An error occurred loading the map");
				});
				
				// Add load handler
				map.current.on('load', () => {
					// console.log("Map load event fired");
					setIsLoaded(true);
					
					try {
						// console.log("Setting fog effect...");
						map.current.setFog({
							'color': 'rgb(186, 210, 235)',
							'high-color': 'rgb(36, 92, 223)',
							'horizon-blend': 0.02,
							'space-color': 'rgb(11, 11, 25)',
							'star-intensity': 0.6
						});
						

						const createEllipse = (center, radiusX, radiusY, rotation = 0, points = 50) => {
							const coords = [];
							for (let i = 0; i < points; i++) {
								const angle = (i / points) * 2 * Math.PI;
								
								// Calculate unrotated coordinates
								const rawX = radiusX * Math.cos(angle);
								const rawY = radiusY * Math.sin(angle);
								
								// Apply rotation
								const rotatedX = rawX * Math.cos(rotation) - rawY * Math.sin(rotation);
								const rotatedY = rawX * Math.sin(rotation) + rawY * Math.cos(rotation);
								
								// Convert to geographic coordinates
								const x = center[0] + rotatedX / Math.cos(center[1] * Math.PI / 180);
								const y = center[1] + rotatedY;
								
								coords.push([x, y]);
							}
							// Close the ring
							coords.push(coords[0]);
							
							return {
								'type': 'Feature',
								'geometry': {
									'type': 'Polygon',
									'coordinates': [coords]
								}
							};
						};
						
						map.current.addSource('prunksaal-ellipse', {
							'type': 'geojson',
							'data': createEllipse(
								[16.36667, 48.20617], // center
								0.00011,              // radiusX in degrees
								0.00009,              // radiusY in degrees
								40 * (Math.PI / 180)      // rotation in radians (0 = pointing up)
							)
						});
						
						map.current.addLayer({
							'id': 'prunksaal-glow-outline',
							'type': 'line',
							'source': 'prunksaal-ellipse',
							'paint': {
								'line-color': '#FF0066',
								'line-opacity': 0.8,
								'line-width': [
									'interpolate',
									['exponential', 1.75],
									['zoom'],
									2, 1,
									15, 10,
									18, 30
								],
								'line-blur': 5
							}
						});
						
						let offset = 0.00005
						
						// console.log("Starting flyTo animation...");
						
						const bounds = [
							[16.36667 - 0.00015, 48.20617 - 0.00015], // Southwest coordinates
							[16.36667 + 0.00015, 48.20617 + 0.00015]  // Northeast coordinates
						];
						
						setTimeout(()=>{
							setIsFlying(true)
						}, 7000)
						
						setTimeout(() => {
							const screenWidth = window.innerWidth;
							const leftPadding = screenWidth*0.37;
							
							map.current.fitBounds(bounds, {
								padding: { top: 50, bottom: 50, left: leftPadding, right: 0 },
								bearing: -127.21,
								pitch: 0,
								duration: 12000
							});
						}, 9000);
					} catch (err) {
						console.error("Error in load handler:", err);
						setError(err.message);
					}
				});
				
			} catch (err) {
				console.error("Error during map initialization:", err);
				setError(err.message);
			}
		};
		
		initializeMap().catch(err => {
			console.error("Caught error during initialization:", err);
			setError(err.message);
		});
		
		return () => {
			if (map.current) {
				map.current.remove();
				map.current = null;
			}
		};
	}, [MAPBOX_TOKEN]);
	
	if (isRemoved) return null;
	
	return (
		<div
			className="fixed inset-0"
			style={{
				position: 'fixed',
				pointerEvents: 'none',
				top: 0,
				left: 0,
				right: 0,
				bottom: 0,
				zIndex: 1000,
				backgroundColor: 'black',
				opacity: isFading ? 0 : 1,
				transition: 'opacity 1s ease-out',
			}}
		>
			{error && (
				<div
					className="absolute top-0 left-0 z-50 p-4 bg-red-500 text-white"
					style={{ zIndex: 1001 }}
				>
					Error: {error}
				</div>
			)}
			
			<div
				ref={mapContainer}
				style={{
					position: 'absolute',
					top: 0,
					left: 0,
					width: '100%',
					height: '100%',
					background: 'transparent'
				}}
			/>
			
			{isLoaded && isFlying && (
				<div className={styles.introOverlay}>
					<div className={styles.introText} style={{marginTop: '2rem'}}>
						<p></p>
						<h2> Folgen Sie uns zum Prunksaal der Wiener Hofburg und tauchen Sie ein in die Welt der “Bibliotheca Eugeniana Digital”.</h2>
					</div>
				</div>
			)}

			{isLoaded && !isFlying && (
				<div className={styles.introOverlay}>
					<div className={styles.introText} style={{marginTop: '2rem'}}>
						<h2>Deutlich mehr zahlte Kaiser Karl VI. für die erlesene Privatbibliothek des Prinzen (150.000 Gulden), die sich heute in der Österreichischen Nationalbibliothek befindet.</h2>
					</div>
				</div>
			)}

			{!isLoaded && (
				<div
					className={styles.introOverlay}
					style={{
						height: '100vh',
						position: 'fixed',
						top: '50%',
						left: '50%',
						paddingTop: '40vw',
						marginBottom: '20vw',
						transform: 'translate(-50%, -50%)',
						backgroundColor: 'rgba(0, 0, 0, 0.9)' // Optional: adds a semi-transparent overlay
					}}
				>
					<div className={styles.introText}>
						<h2>Was war der wertvollste Beitrag des Prinz Eugen von Savoyen zur Kulturgeschichte Wiens?</h2>
						<p></p>
						<h2 style={{fontWeight: '400'}}>Die meisten Besucher der Stadt würden an das Schloss Belvedere denken, das im 18. Jahrhundert 100.000 Gulden wert war.</h2>
					</div>
					<div style={{ margin: '5rem 0 0 0' }}>
						<div className={styles.loader} style={{ margin: '0 auto', padding: '1rem' }} ></div>
					</div>
					<div style={{ textAlign: 'center' }}>... loading ...</div>
				</div>
			)}

				<button
					onClick={handleSkip}
					className={styles.introSkipButton}
					style={{
						visibility: allowSkip ? 'visible' : 'hidden',
					}}
					onMouseOver={(e) => e.target.style.backgroundColor = 'rgba(255, 255, 255, 0.3)'}
					onMouseOut={(e) => e.target.style.backgroundColor = 'rgba(255, 255, 255, 0.2)'}
				>
					Intro überspringen
				</button>

		</div>
	);
};

