import styles from "../styles.module.css";
import React, {useState, useRef, useEffect, useCallback} from 'react';
import {useStore} from '../store';
import {OvalSvg} from "./OvalSvg.jsx";
import {MdStarRate, MdAccessTime} from "react-icons/md";
import {RiBubbleChartLine} from "react-icons/ri";
import {IoMdGlobe} from "react-icons/io";
import {VscComment} from "react-icons/vsc";
import {CustomIconCirclePacking} from "./CustomIcons.jsx";
import {CombinedViewer} from "./Mirador.jsx";
import PhotoAlbum from "react-photo-album";
import Lightbox from "yet-another-react-lightbox";
import Zoom from "yet-another-react-lightbox/plugins/zoom";
import Captions from "yet-another-react-lightbox/plugins/captions";
import {PiCaretLeftBold, PiCaretRightBold} from 'react-icons/pi';
import Inline from "yet-another-react-lightbox/plugins/inline";
import "yet-another-react-lightbox/plugins/captions.css";
import "../assets/lightbox.css";
import "../assets/photoalbum.css";

export const Highlights = () => {
	
	const divRef = useRef(null);
	const welcomeRef = useRef(null);
	
	const activeComponent = useStore(state => state.activeComponent);
	
	const dim = useStore(state => state.dim);
	const showHighlights = useStore(state => state.showHighlights);
	const setShowHighlights = useStore(state => state.setShowHighlights);
	const selectedHighlight = useStore(state => state.selectedHighlight);
	const showCloseup = useStore(state => state.showCloseup);
	const setShowCloseup = useStore(state => state.setShowCloseup);
	const showTour = useStore(state => state.showTour)
	const setShowTour = useStore(state => state.setShowTour)
	const setMode = useStore(state => state.setMode)
	const setShowTutorial = useStore(state => state.setShowTutorial)
	const setShowImpressum = useStore(state => state.setShowImpressum)
	const setChartType = useStore(state => state.setChartType);
	const filterMembership = useStore(state => state.filterMembership)
	const setFilterMembership = useStore(state => state.setFilterMembership)
	const setShowStories = useStore(state => state.setShowStories)
	const setHighlightsOffset = useStore(state => state.setHighlightsOffset);
	
	useEffect(() => {
		if (!showHighlights && showCloseup) {
			setShowCloseup(false)
		}
	}, [showHighlights]);
	
	const handleClick = (e) => {
		let target = e.currentTarget.id;
		
		if (target === 'highlight') {
			return;
			setShowHighlights(!showHighlights);
			if (!showHighlights) {
				setChartType('member');
			}
			return;
		}
		
		if (target === 'memberships' || target === 'colors') {
			// Change Farbschema without affecting highlight mode
			setMode(target);
		} else if (target === 'member') {
			setChartType(target);
			if (showHighlights) {
				setShowHighlights(false);
			}
		} else if (['taxonomyCircle', 'time', 'geomap', 'language'].includes(target)) {
			setChartType(target);
			setShowHighlights(false); // Turn off highlights for these chart types
		} else if (['impressum', 'tutorial', 'tour3d', 'stories'].includes(target)) {
			if (target === 'impressum') setShowImpressum(true);
			if (target === 'tutorial') setShowTutorial(true);
			if (target === 'stories') setShowStories(true);
			if (target === 'tour3d') {
				if (showTour) {
					setShowTour(false)
					return;
				}
				setHighlightsOffset(welcomeRef?.current?.offsetWidth || 0)
				setShowTour(true)
			}
			return;
		}
		if (filterMembership) {
			setFilterMembership(null);
		}
	};
	
	const ratio = dim.screen.width/dim.screen.height;
	const height = dim.screen.height;
	const width = dim.screen.width;
	
	return (<div className={styles.highlightsContainer} style={{width: dim.width, height: dim.left.height}}>
			{!selectedHighlight && (
				<div ref={welcomeRef} className={styles.welcomeContainer}
			        style={{
			            fontSize: ratio > 1.8 ? '0.9rem' : '1.2rem',
			            lineHeight: ratio > 1.8 ? '1.1rem' : '1.4rem',
			            }}
			>
					<p className={styles.welcomeText}>
						<span className={styles.welcomeTextLarge}>W</span>
						illkommen zur Bibliotheca Eugeniana im Prunksaal der Österreichischen Nationalbibliothek
						(rechts) - einer der schönsten Bibliotheken des Barocks, die ein paar der wertvollsten
						Kulturschätze
						der Menschheit enthält.
					</p>
					
					<p className={styles.welcomeText}>Dieses Interface erlaubt die Erkundung und Analyse dieser
						einmaligen
						Sammlung nach verschiedenen Gesichtspunkten:
					</p>
					<div className={styles.iconContainer}>
						<div className={styles.iconItem}>
							<div id={'highlight'} className={styles.icon} onClick={handleClick}
							     style={{border: '2px solid black'}}><MdStarRate size={20}/></div>
							<span onClick={handleClick} className={styles.welcomeTextButton}>was sind Highlights der Sammlung?</span>
						</div>
						<div className={styles.iconItem}>
							<div id={'taxonomyCircle'} className={styles.icon} onClick={handleClick}>
								<CustomIconCirclePacking color='grey' size={24} width={23} height={23}/></div>
							<span id={'taxonomyCircle'} onClick={handleClick} className={styles.welcomeTextButton}>welche Themen sind abgedeckt?</span>
						</div>
						<div className={styles.iconItem}>
							<div id={'geomap'} className={styles.icon} onClick={handleClick}><IoMdGlobe color='grey'
							                                                                            size={20}/>
							</div>
							<span id={'geomap'} onClick={handleClick} className={styles.welcomeTextButton}>woher stammen die Bücher?</span>
						</div>
						<div className={styles.iconItem}>
							<div id={'time'} className={styles.icon} onClick={handleClick}><MdAccessTime color='grey'
							                                                                             size={20}/>
							</div>
							<span id={'time'} onClick={handleClick} className={styles.welcomeTextButton}>wann wurden sie geschrieben?</span>
						</div>
						<div className={styles.iconItem}>
							<div id={'language'} className={styles.icon} onClick={handleClick}><VscComment color='grey'
							                                                                               size={18}/>
							</div>
							<span id={'language'} onClick={handleClick} className={styles.welcomeTextButton}>in welcher Sprache?</span>
						</div>
						<div className={styles.iconItem}>
							<div id={'member'} className={styles.icon} onClick={handleClick}><RiBubbleChartLine
								color='grey' size={20}/></div>
							<span id={'member'} onClick={handleClick} className={styles.welcomeTextButton}>was gehört überhaupt dazu?</span>
						</div>
					</div>
					
					<p className={styles.welcomeText}>Sie können gleich hier Highlights erkunden
						(siehe <MdStarRate size={16} style={{verticalAlign: 'middle'}}/>), oder andere
						Gesichtspunkte probieren - siehe auch die Buttons oben rechts.
					</p>
					<p className={styles.welcomeText}>
                        <span className={styles.welcomeButtonContainer}>
                            <span className={styles.welcomeButton} id="stories" onClick={handleClick}>
                                Story
                            </span>
                            <span className={styles.buttonSeparator}>,</span>
                            <span className={styles.welcomeButton}
                                  style={{
	                                  borderWidth: showTour ? '3px' : '2px',
	                                  boxShadow: showTour ? '0 2px 4px rgba(0, 0, 0, 0.3)' : 'none',
                                  }}
                                  id="tour3d" onClick={handleClick}>
                                3D-Rundgang
                            </span>
                            <span className={styles.buttonSeparator}>,</span>
                            <span className={styles.welcomeButton} id="tutorial" onClick={handleClick}>
                                Tutorial
                            </span>
                            <span className={styles.buttonSeparator}>&</span>
                            <span className={styles.welcomeButton} id="impressum" onClick={handleClick}>
                                Impressum
                            </span>
                        </span>
						{' '}bieten Ihnen weitere Hintergrundinformationen.
					</p>
				</div>)}
			{selectedHighlight && (<div ref={divRef} className={styles.highlightDescription}>
					<HighlightDescription offset={divRef?.current?.offsetWidth}/>
				</div>)}
			{activeComponent === 'combined-viewer' && showCloseup && (<div>
					<CombinedViewer
						identifier={showCloseup}
						offset={divRef?.current?.offsetWidth - dim.left.width || 0}
					/>
				</div>)}
			<div className={styles.ovalContainer}>
				<div className={styles.ovalInner}>
					<OvalSvg/>
				</div>
			</div>
		</div>);
}

export const HighlightDescription = ({offset}) => {
	
	const selectedHighlight = useStore(state => state.selectedHighlight);
	
	if (!selectedHighlight) return null;
	
	const highlightTitles = {
		1: "Der Magnetismus der Musik",
		2: 'Luthers Bibel „letzter Hand“',
		3: "Das Hauptwerk Isaac Newtons",
		4: "Der Spion & die Erdbeere",
		5: "Dante's göttliche Komödie im Kleinformat",
		6: "Der Knabe mit dem Elefantenkopf",
		7: "Die Laster & Tugenden der Tiere",
		8: "Prinz Eugen und Gottfried Wilhelm Leibniz",
		9: "Der anatomische Atlas",
		10: "Eines der ersten (europäischen) naturkundlichen Bücher über China",
	};
	
	return (<div className={styles.highlightDescriptionContainer}>
			<div className={styles.highlightTitle}>{highlightTitles[selectedHighlight]}</div>
			<div className={styles.highlightContent}>
				<HighlightContent id={selectedHighlight} offset={offset}/>
			</div>
		</div>);
};

const OenbDigital = ({identifier, children}) => {
	
	const showCloseup = useStore(state => state.showCloseup);
	const setShowCloseup = useStore(state => state.setShowCloseup);
	
	return (<span className={styles.HighlightsLink}
	              onClick={() => setShowCloseup(!showCloseup ? identifier : false)}>{children}</span>)
	
}

export const HighlightContent = ({id, offset}) => {
	
	const imagesByHighlight = {
		1: [
			{src: "highlights/Kirchner_1.png", width: 892, height: 1152},
			{src: "highlights/Kirchner_2.png", width: 606, height: 840},
			{src: "highlights/Kirchner_3.png", width: 840, height: 752},
			{src: "highlights/Kirchner_4.png", width: 826, height: 1080},
			{src: "highlights/kirchner_5.png", width: 1012, height: 1310},
		],
		2: [
			{src: "highlights/luther1.png", width: 1026, height: 1484},
			{src: "highlights/Luther2.png", width: 978, height: 1416},
			{src: "highlights/luther3.png", width: 1034, height: 1530},
			{src: "highlights/luther4.png", width: 1400, height: 1315},
			{src: "highlights/luther5.png", width: 1032, height: 758},
			{src: "highlights/luther6.png", width: 1010, height: 976},
		],
		3: [
			{src: "highlights/Newton_1.png", width: 906, height: 1008},
			{src: "highlights/Newton_2.png", width: 984, height: 1302},
			{src: "highlights/Newton_3.png", width: 920, height: 684},
			{src: "highlights/Newton_4.png", width: 916, height: 646},
			{src: "highlights/Newton_5.png", width: 712, height: 804},
		],
		4: [
			{src: "highlights/Frezier_1.png", width: 848, height: 1112},
			{src: "highlights/Frezier_2.png", width: 1800, height: 1226},
			{src: "highlights/Frezier_3.png", width: 806, height: 1032},
			{src: "highlights/Frezier_4.png", width: 928, height: 1166},
			{src: "highlights/Frezier_5.png", width: 894, height: 1160},
		],
		5: [
			{src: "highlights/dante_1.png", width: 810, height: 668},
			{src: "highlights/dante_2.png", width: 852, height: 850},
			{src: "highlights/dante_3.png", width: 606, height: 890},
			{src: "highlights/dante_4.png", width: 898, height: 1475},
			{src: "highlights/dante_5.png", width: 706, height: 546},
		],
		6: [
			{src: "highlights/Aldrovandi_1.png", width: 695, height: 1134},
			{src: "highlights/Aldrovandi_2.png", width: 734, height: 1162},
			{src: "highlights/Aldrovandi_3.png", width: 780, height: 1116},
			{src: "highlights/Aldrovandi_4.png", width: 636, height: 1102},
			{src: "highlights/Aldrovandi_5.png", width: 1348, height: 754},
			{src: "highlights/Aldrovandi_6.png", width: 1022, height: 800},
		],
		7: [
			{src: "highlights/gessner_1.png", width: 825, height: 608},
			{src: "highlights/gessner_2.png", width: 806, height: 1112},
			{src: "highlights/gessner_3.png", width: 818, height: 996},
			{src: "highlights/gessner_4.png", width: 796, height: 720},
			{src: "highlights/gessner_5.png", width: 1256, height: 756},
			{src: "highlights/gessner_6.png", width: 1546, height: 1040},
			{src: "highlights/gessner_7.png", width: 1176, height: 838},
			{src: "highlights/gessner_8.png", width: 1100, height: 876},
			{src: "highlights/gessner_9.png", width: 796, height: 1212},
		],
		8: [
			{src: "highlights/leibniz_1.png", width: 996, height: 1360},
			{src: "highlights/leibniz_2.png", width: 878, height: 444},
			{src: "highlights/leibniz_3.png", width: 818, height: 360},
			{src: "highlights/leibniz_4.png", width: 828, height: 1136},
		],
		9: [
			{src: "highlights/vesali_1.png", width: 1304, height: 1668},
			{src: "highlights/vesali_2.png", width: 710, height: 1230},
			{src: "highlights/vesali_3.png", width: 750, height: 1204},
			{src: "highlights/vesali_4.png", width: 684, height: 1180},
			{src: "highlights/vesali_5.png", width: 666, height: 1158},
			{src: "highlights/vesali_6.png", width: 768, height: 578},
			{src: "highlights/vesali_7.png", width: 668, height: 1184},
		],
		10: [
			{src: "highlights/Boym_1.png", width: 1510, height: 1848},
			{src: "highlights/Boym_2.png", width: 1516, height: 1848},
			{src: "highlights/Boym_3.png", width: 1848, height: 1524},
			{src: "highlights/Boym_4.png", width: 1454, height: 1848},
			{src: "highlights/Boym_5.png", width: 912, height: 1402},
		],
	};
	
	const images = imagesByHighlight[id] || [];
	
	switch (id) {
		case 1:
			return (<>
				<p>Das Werk des deutschen Jesuiten und Universalgelehrten Athanasius Kircher (1602-1680) enthält ein
					Kapitel über den <strong>Magnetismus der Musik</strong>. Zentrales Thema des Buches ist Kirchers
					Begriff des Magnetismus in den unterschiedlichsten Bereichen der Natur -- bis hin zur scharfen
					Kritik an Keplers Theorie des kosmischen Magnetismus.</p>
				
				<p>Er beschreibt darin physikalische Experimente, behandelt aber auch die Wirkung der <strong>Tarantella
					als Gegengift zum Gift der Tarantel</strong>: Wer von dieser Spinne gebissen wurde, sollte zur
					schnellen Musik der Tarantella bis zur Erschöpfung tanzen, um das Gift aus dem Körper zu
					treiben.</p>
				
				<p>
					Erkunden Sie dieses Buch <OenbDigital identifier={'Z181596601'}>hier</OenbDigital>.
				</p>
				
				
				<p><em>Kircher, Athanasius</em><br/>
					Magnes Siue De Arte Magnetica Opvs Tripartitvm<br/>
					Köln: Apud Iodocvm Kalcoven, 1643.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z181596601'}>BE.3.O.63 Alt Prunk</OenbDigital></p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 2:
			return (<>
				<p>Die Übersetzung der Bibel ins Deutsche ist Martin Luthers (1483-1546) bedeutendes literarisches
					Vermächtnis. Die letzte zu seinen Lebzeiten gedruckte Ausgabe dieser Bibel, die sogenannte
					„Ausgabe letzter Hand" von 1545, gilt nach Luthers Tod als maßgeblich. Sie stellt den Höhepunkt
					und auch den Endpunkt der von Luther autorisierten Bibelausgaben dar.</p>
				
				<p>Typographie und Abbildungen der beiden Foliobände sind von hervorragender Qualität. Die <strong>handkolorierten
					Holzschnitt-Illustrationen</strong> haben bis heute nichts an ihrer Wirkkraft verloren.
					Besonders interessant sind die Darstellungen zur Offenbarung des Johannes (Apokalypse), z.B. die
					Hure Babylon mit dreifacher Krone, einer Anspielung auf die päpstliche Tiara -- sie steht hier
					für die katholische Kirche, die Darstellung quasi ein Kampfsymbol der Reformation.</p>
				
				<p>
					<OenbDigital identifier={'Z16538530X'}>Band 1</OenbDigital>
				</p>
				<p>
					<OenbDigital identifier={'Z165387204'}>Band 2</OenbDigital>
				</p>
				
				<p><em>Luther, Martin</em><br/>
					Biblia: das ist: Die gantze Heilige Schrifft: Deudsch.<br/>
					Wittenberg: durch Hans Lufft, 1544-1545.<br/>
					ÖNB-Signatur: BE.1.G.8-9 Alt Prunk
				</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 3:
			return (<>
				<p>Die mathematischen Grundprobleme, die Isaac Newton (1643-1727) in seiner <OenbDigital
					identifier={'Z182029301'}>Principia</OenbDigital> beschreibt, hatte er bereits als 24jähriger
					Student gelöst.
					Veröffentlicht wurde das Werk jedoch erst 20 Jahre später, als der schon berühmte Mathematiker
					und Physiker für die Royal Society die Ergebnisse seiner Berechnungen zu Fragen der universellen
					Gravitation und ihrer Anwendung auf das Weltsystem darlegte.
				</p>
				
				<p>Von dieser seltenen dritten Ausgabe wurden nur 200 Stück verlegt. Sie war die letzte, die zu
					Lebzeiten des Autors veröffentlicht wurde, und die Grundlage aller nachfolgenden Ausgaben.</p>
				
				<p>Das vorliegende Exemplar trägt die <strong>eigenhändige Widmung</strong> des Vize-Präsidenten der
					Royal Society, Richard Mead
					 {/*<ExternalLink href="https://de.wikipedia.org/wiki/Richard_Mead">Richard*/}
						{/*Mead</ExternalLink>*/}
						, der ein
					Bewunderer Eugens war und dem Prinzen mehrere Werke dedizierte. Der Band behielt seinen
					ursprünglichen Einband, wie dies auch bei anderen Widmungsbänden in der Bibliothek Eugens
					festgestellt werden kann.</p>
				
				<p><em>Newton, Isaac</em><br/>
					Philososphiae Naturalis Principia Mathematica.<br/>
					London: Apud Guil. & Joh. Innys, 1726.<br/>
					ÖNB-Signatur: 79.E.39 Alt Prunk</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 4:
			return (<>
				<p><strong>Erdbeeren</strong> waren in Europa schon seit Jahrhunderten bekannt, aber vor allem die
					Walderdbeere. Die große moderne Erdbeere haben wir einem französischen <strong>Spion</strong> zu
					verdanken, dem Militäringenieur Amédée-François Frézier (1682-1773).</p>
				
				<p>Im Auftrag des französischen Königs Ludwig XIV. kartierte er in Südamerika die spanischen
					Militäroperationen. Ludwig erhob Anspruch auf den spanischen Thron und wollte die Stärke des
					spanischen Militärs erkunden. Er erteilte Frézier noch einen anderen Auftrag: Ein früher
					entsandter Spion hatte von großen schmackhaften Erdbeeren erzählt, die in Peru wachsen sollten,
					und diese Erdbeeren wollte Ludwig gerne haben. Frézier reiste als Kaufmann, ein guter Deckmantel
					und ein Alibi, um für alles Interesse zu zeigen, auch für Obst. Er fand die Erdbeeren und auf
					der Rückreise nahm er mehrere Erdbeerpflanzen mit.</p>
				
				<p>Seinen Reisebericht - inkl. vieler militärischer und botanischer Skizzen - können Sie in der
					Bibliotheca Eugeniana erkunden: <strong>
						<OenbDigital identifier={'Z220707000'}>Hier</OenbDigital></strong> geht
					es zur Ansicht des Objekts.</p>
				
				<p><em>Frézier, Amédée François:</em><br/>
					Relation Du Voyage De La Mer Du Sud Aux Côtes Du Chily Et Du Perou<br/>
					Paris: Chez Jean-Geoffroy Nyon, 1716.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z220707000'}>BE.9.L.3 Alt Prunk</OenbDigital>
				</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"  // This will create widescreen thumbnails
					lightboxWidth="auto" // This sets the width of the lightbox
					carouselWidth="auto"
					offset={offset}
				/>
			</>);
		case 5:
			return (<>
				<p>Zahlreiche große Druckausgaben von Dante Alighieris großem theologischen Gedicht waren bereits
					seit fast dreißig Jahren im Umlauf, als Aldus Manutius im Jahr 1502 zum ersten Mal eine
					kleinformatige Oktavausgabe des Gedichts druckte. Er verzichtete auf Kommentare und Textzusätze,
					um eine erschwingliche, leicht zu transportierende und sehr einfache Version des Textes
					für <strong>Buchliebhaber</strong> zu bieten. Aldus hat bei der Vorbereitung dieser Ausgabe mit
					seinem Freund, dem italienischen, humanistischen Gelehrten,
					Dichter und Kardinal Pietro Bembo
					zusammengearbeitet und ein frühes Manuskript der Commedia als Grundlage genutzt. Dieser Text
					wurde in den folgenden 300 Jahren zur Standardversion für Dante-Ausgaben.</p>
				
				<p>In der Bibliotheca Eugeniana befindet sich eine handkolorierte Pergamentausgabe mit braunem
					Maroquineinband aus dieser Druckserie. Erkunden Sie diese <OenbDigital
						identifier={'Z221957802'}>hier</OenbDigital>.</p>
				
				<p>Der Typograph Manutius und sein hochtalentierter Schriftschneider Francesco Griffo aus Bologna
					schufen für ihre Aldinen die erste kursive
					Drucktype aus Blei. Der Druck mit diesen schmallaufenden und damit platzsparenden Typen
					revolutionierte den Buchdruck und leitete die Entwicklung
					der <strong>Taschenbücher</strong> ein.</p>
				
				<p><em>Dante &lt;Alighieri&gt;, 1265-1321</em><br/>
					Le Terze Rime<br/>
					Einheitssachtitel: La divina commedia<br/>
					Venedig: Aldus Manutius, 1502.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z221957802'}>22.O.28 Alt Prunk</OenbDigital>
				</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 6:
			return (<>
				<p>"Puer capite elephantino", ein Knabe mit Elefantenkopf -- Solche Missbildungen, also <strong>Menschen
					mit Tierköpfen</strong>, seien sehr selten, doch könne man bereits in der Geschichte des
					Virgilius Polydorus Nachrichten darüber finden. Und glaubwürdige Männer hätten berichtet, dass
					in Ungarn ein Knabe mit Elefantenkopf zur Welt gekommen sei.</p>
				
				<p>Ulisse Aldrovandi (1522-1605) beschrieb In seiner <strong>monstrorum historia</strong> (Historie
					der Monster) neben Missbildungen bei Menschen, Tieren und Pflanzen auch eine Reihe von
					Fabelwesen, die zu seiner Zeit als existent galten.</p>
				
				<p>Aldrovandi studierte in seiner Heimatstadt Bologna zuerst Mathematik, später auch Logik, dann
					Medizin in Padua und Bologna. Dort gründete er 1568 einen der ersten botanischen Gärten. Neben
					seiner Historie der Monster verfasste zahlreiche Schriften über Pflanzen, Tiere (Historia
					animalium in 11 Bänden) und Mineralien.</p>
				
				<p><em>Aldrovandi, Ulisse</em><br/>
					Monstrorum historia<br/>
					Bologna: Typis Nicolai Tebaldini, 1642.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z197817901'}>BE.4.G.23 Alt Prunk</OenbDigital>
				</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
				
			</>);
		case 7:
			return (<>
				<p>Im 16. Jahrhundert war die Tierkunde eine weitgehend unerforschte Disziplin. Der Zürcher
					Stadtarzt <strong>Conrad Gessner</strong> (1516-1565) versuchte als Erster, die Tiere aller
					damals bekannten Kontinente zu beschreiben. Er schuf eine gewaltige <OenbDigital
						identifier={'Z175515802'}>Tierenzyklopädie</OenbDigital> mit über 1.000 Tieren in Wort und Bild.
					Gessner trug
					das damalige und antike Wissen zusammen, ordnete es neu und prägte damit die Tierkunde für
					Generationen auf dem langen Weg zur modernen Zoologie.</p>
				
				<p>In seiner Historia animalium porträtierte er die Tiere anhand unterschiedlichster Facetten: Neben
					Lebensraum, Lebensweise und Physiologie beschrieb er auch deren gottgegebenen Nutzen, sowie
					ihre <strong>Laster und Tugenden</strong> -- etwa den selbstlosen Pelikan. Als belesener
					Universalgelehrter erläuterte Gessner in seiner Enzyklopädie auch die vielschichtigen
					Tierdeutungen in Literatur und Kunst, in Sprichwörtern und Fabeln. In der Historia animalium
					sind einige Tiere aufgeführt, die heute als Fabeltiere gelten, etwa das <strong>Einhorn</strong>.
				</p>
				
				<p>Die Qualität der zahlreichen Holzschnitte unterschied seine Veröffentlichung von denen seiner
					Fachkollegen. Neben eigenen Illustrationen, verwendete Gessner Bilder von Zeitgenossen, so auch
					das weltberühmte Rhinocerus von Albrecht Dürer sowie die Giraffe aus Bernhard von Breydenbachs
					Peregrinatio in terram sanctam.</p>
				
				<p><em>Gessner, Conrad</em><br/>
					Historia animalium.<br/>
					Frankfurt, Main: In Bibliopolio Cambieriano, 1603.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z175515802'}>BE.4.F.26-28 Alt Prunk</OenbDigital>
				</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 8:
			return (<>
				<p>Als Förderer der Wissenschaften bemühte sich Philosoph Gottfried Wilhelm Leibniz (1646-1716) auch
					in Wien um die Gründung einer Akademie. In Wien traf er auch mit Prinz Eugen zusammen, der
					Leibniz' Idee befürwortete.</p>
				
				<p>Leibniz legte seine <strong>philosophische Doktrin</strong> nie in einem zusammenfassenden Werk
					dar, sondern nur in Teilen in zahlreichen Briefen und Abhandlungen (teilweise auch
					veröffentlicht). Prinz Eugen erbat sich von ihm daher eine allgemein verständliche Einführung in
					seine Lehre, dessen Ergebnis der Aufsatz „Principes de la Nature et de la Grace" war.</p>
				
				<p>Erst 1718, also nach Leibniz Tod, wurde der Aufsatz in lateinischer Übersetzung unter dem Titel
					„Principia philosophiae" veröffentlicht. Diese Übersetzung stellte der Philosoph Michael
					Gottlieb Hansch (1683-1749) seiner Leibnitii Principia Philosophiae (1728) voran, die er Prinz
					Eugen widmete. Hansch stellte darin aus den bis dato bekannten und zusammenhanglosen Texten
					Leibniz' und den mit ihm gewechselten Briefen eine systematische und authentische Darstellung
					von dessen Metaphysik vor. Die Eigenleistung Hanschs ist hierbei kaum zu überschätzen. Nicht
					allein einschlägige Texte Leibniz' werden zur Veranschaulichung einzelner Punkte herangezogen,
					sondern auch solche aus allen Epochen der Geschichte der Philosophie.</p>
				
				<p>Hansch machte während seiner Studienzeit in Leipzig Bekanntschaft mit dem Philosophen Leibniz,
					stand später mit ihm in Briefwechsel und begegnete ihm in den Jahren 1712 bis 1714 in Wien
					wieder.</p>
				
				<p>Prinz Eugen verwahrte einige weitere Werke von Leibniz in seiner Bibliothek.</p>
				
				<p><em>Leibniz, Gottfried Wilhelm</em><br/>
					Principia Philosophiæ, More Geometrico Demonstrata.<br/>
					Frankfurt, Leipzig: Petrus Conradus Monath, 1728.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z197997708'}>BE.5.N.45 Alt Prunk</OenbDigital></p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 9:
			return (<>
				<p>Der 1543 veröffentlichte Anatomieatlas von Andreas Vesalius "De humani corporis fabrica" (Über
					den <strong>Bau des menschlichen Körpers</strong>) ist wohl der berühmteste medizinische Atlas
					der Medizingeschichte. Er ist zugleich ein Glanzstück und Meisterwerk der Buchkunst seiner Zeit
					und ein Klassiker der Medizin. Berühmt wurde es vor allem auch wegen seiner neuartigen,
					künstlerisch hochwertigen Illustrationen, welche von Ganzfiguren bis zu kleinsten Strukturen,
					wie etwa den Gehörknöchelchen, reichen. <OenbDigital
						identifier={'Z155287209'}>Hier</OenbDigital> können
					Sie diese selbst erkunden.</p>
				
				<p>Andreas Vesalius (1514-1564) war Leibarzt Kaiser Karl V. und gilt als der <strong>Begründer der
					modernen Anatomie</strong>. Er sezierte nicht in der bisher üblichen Weise, um an ein bestimmtes
					Organ zu gelangen, sondern er legte den Körper Schicht für Schicht frei. Damit konnte er als
					erster auch den gesamten Blutkreislauf exakt darstellen. Der gebürtige Flame, der in Löwen,
					Paris und Padua Medizin studierte, war kritisch und innovativ in seinen Methoden und ersetzte
					Autoritätsglauben durch Empirie.</p>
				
				<p><em>Vesalius, Andreas,</em><br/>
					Scholae medicorum Patauinæ professoris, de Humani corporis fabrica<br/>
					Basel: Ex officina Ioannis Oporini, 1543.<br/>
					ÖNB-Signatur: <OenbDigital identifier={'Z155287209'}>BE.5.E.5 Alt Prunk</OenbDigital></p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		case 10:
			return (<>
				<p>Michał Piotr Boym (1612-1659) war ein polnischer Wissenschaftler, Entdecker, <ExternalLink
					href="https://de.wikipedia.org/wiki/Jesuiten">Jesuit</ExternalLink> und Missionar in China. Er
					war einer
					der ersten Europäer, der das zentrale China bereiste und verfasste zahlreiche Bücher über
					die <strong>asiatische Fauna, Flora und Geografie</strong>.</p>
				
				<p>Sein berühmtestes Werk, die <OenbDigital identifier={'Z175515607'}>Flora
					Sinensis</OenbDigital> (Chinesische Flora), wurde
					während eines Europaaufenthaltes von den Jesuiten in Wien bei der kleinen, jedoch
					hochqualifizierten und bereits bei anderen Drucken aus der Chinamission bewährten Offizin des
					Matthäus Rickhes in Auftrag gegeben. Neben der anschaulichen Darstellung der Pflanzen enthält
					das Buch auch, was im Titel nicht zum Ausdruck kommt, <strong>Darstellungen aus der
						Tierwelt</strong>. Mit 21 Pflanzen und 8 Tieren stellt Boym nur die, wie er sagt,
					hauptsächlichsten Früchte, Bäume und Tiere der Chinesen und Inder dar.</p>
				
				<p>Das Werk ist dem künftigen Kaiser Leopold I., als dem damaligen König von Ungarn, gewidmet und
					enthält auch Appelle zur Unterstützung
					des katholischen chinesischen Kaisers.</p>
				
				<p><em>Boym, Michał Piotr</em><br/>
					Flora Sinensis, fructus floresque humillime porrigens ...<br/>
					Wien: Typis Matthaei Rictij, 1656.<br/>
					ÖNB-Signatur: BE.4.F.21 Alt Prunk</p>
				
				<ThumbnailGridGallery
					images={images}
					columns={3}
					spacing={10}
					aspectRatio="3:4"
					lightboxWidth="auto"
					carouselWidth="auto"
					offset={offset}
				/>
			
			</>);
		default:
			return <p>Content for this highlight is not available.</p>;
	}
};


export const ExternalLink = ({href, children}) => {
	const activeComponent = useStore(state => state.activeComponent);
	const setActiveComponent = useStore(state => state.setActiveComponent);
	
	const toggleIframe = (e) => {
		e.preventDefault();
		if (activeComponent === `external-${href}`) {
			setActiveComponent(null);
		} else {
			setActiveComponent(`external-${href}`);
		}
	};
	
	return (<span>
      <a href={href} onClick={toggleIframe}>{children}</a>
			{activeComponent === `external-${href}` && (<div style={{
					position: 'fixed',
					top: '10%',
					left: '10%',
					width: '80%',
					height: '85%',
					backgroundColor: 'white',
					zIndex: 10,
					boxShadow: '0 0 10px rgba(0,0,0,0.5)',
					display: 'flex',
					flexDirection: 'column'
				}}>
					<div style={{
						display: 'flex', justifyContent: 'space-between', padding: '10px', backgroundColor: '#f0f0f0'
					}}>
						<h3>Externer Content: {children}</h3>
						<button onClick={() => setActiveComponent(null)}>Schließen</button>
					</div>
					<iframe
						src={href}
						style={{
							width: '100%', height: '100%', border: 'none'
						}}
						title={`External Content: ${children}`}
					/>
				</div>)}
    </span>);
};


const ThumbnailGridGallery = ({images, columns = 3, carouselWidth = '60%', offset = 0, spacing = 10}) => {
	
	const dim = useStore(state => state.dim);
	const [index, setIndex] = useState(-1);
	const activeComponent = useStore(state => state.activeComponent);
	const setActiveComponent = useStore(state => state.setActiveComponent);
	
	const handleThumbnailClick = useCallback(({ index }) => {
		setIndex(index);
		setActiveComponent('thumbnail-gallery');
	}, [setActiveComponent]);
	
	const handleClose = useCallback(() => {
		setIndex(-1);
		setActiveComponent(null);
	}, [setActiveComponent]);
	
	// Prepare photos for react-photo-album
	const photos = images.map((image) => ({
		src: image.src,
		width: image.width,
		height: image.height,
		alt: image.title || '',
	}));
	
	const carouselContainerStyle = {
		position: 'absolute',
		top: 0,
		left: `${offset}px`,
		width: carouselWidth === 'auto' ? `${dim.screen.width - offset - 20}px` : carouselWidth,
		height: `${dim.left.height - 30}px`,
		display: activeComponent === 'thumbnail-gallery' ? 'block' : 'none',
		zIndex: 1000,
	};
	
	return (
		<div>
			<PhotoAlbum
				layout="columns"
				photos={photos}
				columns={columns}
				onClick={handleThumbnailClick}
				spacing={spacing}
				targetRowHeight={180}
			/>
			<br />
			{activeComponent === 'thumbnail-gallery' && (
				<div style={carouselContainerStyle}>
					<div className={styles.closeMirador} onClick={handleClose} style={{right: '-10px'}}>
						&#10006;
					</div>
					<Lightbox
						index={index}
						slides={images}
						plugins={[Inline, Captions, Zoom]}
						on={{
							view: ({index}) => setIndex(index),
						}}
						carousel={{
							padding: 0,
							spacing: 0,
						}}
						inline={{
							style: {
								width: '100%',
								height: '100%',
								margin: '0 auto',
							}
						}}
						styles={{
							container: {backgroundColor: "rgba(0, 0, 0, .8)"},
							root: {
								'--yarl__color_backdrop': 'transparent',
								'--yarl__button_root_size': '40px',
								'--yarl__button_icon_size': '24px',
								'--yarl__button_root_background_color': 'rgba(255, 255, 255, 0.2)',
								'--yarl__button_root_hover_background_color': 'rgba(255, 255, 255, 0.4)',
							},
							button: {
								filter: 'none',
								opacity: 0.7,
								padding: '0 5px 0 5px',
								transition: 'opacity 0.3s',
								'&:hover, &:focus': {
									opacity: 1,
								},
								'&:focus:not(:focus-visible)': {
									outline: 'none',
								},
							},
							buttonPrev: {
								marginLeft: '20px',
							},
							buttonNext: {
								marginRight: '20px',
							},
							iconButton: {
								filter: 'none',
								opacity: 0.7,
								padding: 0,
								transition: 'opacity 0.3s',
								'&:hover, &:focus': {
									opacity: 1,
								},
								'&:focus:not(:focus-visible)': {
									outline: 'none',
								},
							},
						}}
						render={{
							iconPrev: () => <PiCaretLeftBold size="1.8em"/>,
							iconNext: () => <PiCaretRightBold size="1.8em"/>,
						}}
					/>
				</div>
			)}
		</div>
	);
};
