/* eslint-disable react/prop-types */

import { arrayRange } from "../lib/array-range";
import styles from "../styles.module.css";

export const FloorPlan = (props) => {

    const {width, height, side} = props;

    const halfWidth = width/2;
    const halfHeight = height/2;
    const halfSide = side/2;

    // oval
    const ovalScaleX = 0.9;
    const ovalScaleY = 1.1;

    // center circle
    const centerCircleScale = 0.2;
    const textArcRadiusScale = 0.15;


    // concentric circles
    const radiusScales = arrayRange(3,10,1).map(i => i / 10);

    // radial lines
    const radialLineRotations = arrayRange(0,170,10);

    // tiles
    const tileRotations = arrayRange(0,340,20)
    const tileTranslateScales = arrayRange(4,10,2).map(i => i / 10);

    return (
        <>
            <defs>
                <mask id="ovalMask">
                    <ellipse
                        cx={halfWidth}
                        cy={halfHeight}
                        rx={halfSide * ovalScaleX}
                        ry={halfSide * ovalScaleY }
                        fill="white"
                    />
                    <circle
                        cx={halfWidth}
                        cy={halfHeight}
                        r={halfSide * centerCircleScale}
                        fill="black"
                    />
                </mask>

                <path
                    id="upperCurve"
                    d={`M ${halfWidth - halfSide * textArcRadiusScale} ${halfHeight} A  ${halfSide * textArcRadiusScale}  ${halfSide * textArcRadiusScale} 0 0 1 ${halfWidth + halfSide * textArcRadiusScale} ${halfHeight}`}
                />
                <path
                    id="lowerCurve"
                    d={`M ${halfWidth - halfSide * textArcRadiusScale} ${halfHeight} A  ${halfSide * textArcRadiusScale}  ${halfSide * textArcRadiusScale} 0 0 0 ${halfWidth + halfSide * textArcRadiusScale} ${halfHeight}`}
                />

                <filter id="whiteOutlineEffect" colorInterpolationFilters="sRGB">
                    <feMorphology in="SourceAlpha" result="MORPH" operator="dilate" radius="2" />
                    <feColorMatrix in="MORPH" result="WHITENED" type="matrix" values="-1 0 0 0 1, 0 -1 0 0 1, 0 0 -1 0 1, 0 0 0 1 0"/>
                    <feMerge>
                        <feMergeNode in="WHITENED"/>
                        <feMergeNode in="SourceGraphic"/>
                    </feMerge>
                </filter>

            </defs>

            <g id="floorPlan">

                <g mask="url(#ovalMask)">

                    {/* Concentric Circles */}
                    <g className={styles.concentricCircle}>
                        {radiusScales.map((radiusScale) => {
                            return (
                                <circle
                                    key={`concentric-circle-${radiusScale}}`}
                                    cx={halfWidth}
                                    cy={halfHeight}
                                    r={halfSide * radiusScale}
                                />
                            )
                        })}
                    </g>

                    {/* Radial Lines */}
                    <g>
                        {radialLineRotations.map((rot) => {
                            return(
                                <rect
                                    key={`radial-line-${rot}`}
                                    x={halfWidth - side * 0.001}
                                    y="0"
                                    width={side * 0.002}
                                    height={height}
                                    transform={`rotate(${rot})`}
                                    className={styles.radialLine}
                                />
                            )
                        })}
                    </g>

                    {/* Tiles */}
                    {tileRotations.map((rot) => {
                        return(
                            <g key={`tiles-${rot}`}>
                                {tileTranslateScales.map((scale) => {
                                    return (
                                        <rect
                                            key={`tiles-${rot}-${scale}`}
                                            x={width/2-side*0.02}
                                            y={height/2-side*0.02}
                                            width={side*0.04}
                                            height={side*0.04}
                                            transform={`rotate(${rot}) translate(0 ${-halfSide * scale}) scale(${scale + 0.2} ${scale + 0.1}) rotate(45)`}
                                            className={styles.radialLine}
                                        />
                                    )
                                })}
                            </g>
                        )
                    })}


                </g>

                {/* Center Circle */}
                <g>
                    <circle
                        cx={halfWidth}
                        cy={halfHeight}
                        r={halfSide * centerCircleScale}
                        strokeWidth={halfSide * 0.01}
                        className={styles.centerCircle}
                    />
                    <image
                        href="statue.png"
                        height={side / 2 * 0.24}
                        width={side / 2 * 0.4}
                        x={width / 2 - side / 2 * 0.2}
                        y={height / 2 - side / 2 * 0.12}
                        className={styles.statue}
                    />
                    <text
                        textAnchor='middle'
                        className={styles.ovalTitle}
                        fontSize={side * 0.02}
                        filter="url(#whiteOutlineEffect)"
                    >
                        <textPath
                            xlinkHref="#upperCurve"
                            startOffset={'50%'}
                        >
                            Bibliotheca Eugeniana
                        </textPath>
                        <textPath
                            xlinkHref="#lowerCurve"
                            startOffset={'50%'}
                            alignmentBaseline='mathematical'
                        >
                            Digital
                        </textPath>
                    </text>
                
                </g>
                
                {/* Outer Oval */}
                <ellipse
                    cx={halfWidth}
                    cy={halfHeight}
                    rx={halfSide * ovalScaleX}
                    ry={halfSide * ovalScaleY}
                    strokeWidth={halfSide * 0.015}
                    className={styles.centerCircle}
                />
                
                {/* Locations */}
                <text
                    x={halfWidth}
                    y={halfHeight - halfSide * ovalScaleY}
                    textAnchor='middle'
                    className={styles.ovalTitle}
                    fontSize={side * 0.018}
                    filter="url(#whiteOutlineEffect)"
                >
                    <tspan alignmentBaseline='middle'>
                        Burggarten
                    </tspan>
                </text>
                <text
                    x={halfWidth}
                    y={halfHeight + halfSide * ovalScaleY}
                    textAnchor='middle'
                    className={styles.ovalTitle}
                    fontSize={side * 0.018}
                    filter="url(#whiteOutlineEffect)"
                >
                    <tspan alignmentBaseline='middle'>
                        Josefsplatz
                    </tspan>
                </text>

            </g>
        </>
    )
}
