import {useState, useEffect} from 'react';
import {useMeasure} from "react-use";
import styles from "../styles.module.css";
import {useStore} from '../store';
import {createColorCategories} from '../lib/categories.jsx';

export const ColorView = () => {
	const [ref, {width, height}] = useMeasure();
	
	const [side, setSide] = useState(Math.min(width, height));
	
	const isFirefox = useStore(state => state.isFirefox);
	
	const filterMembership = useStore(state => state.filterMembership);
	const setFilterMembership = useStore(state => state.setFilterMembership);
	
	const hoverShelf = useStore(state => state.hoverShelf);
	const storedSubCats = useStore(state => state.storedSubCats);
	
	const setHoveredBook = useStore(state => state.setHoveredBook);
	
	const hoveredColor = useStore(state => state.hoveredColor);
	const setHoveredColor = useStore(state => state.setHoveredColor);
	
	const stats = useStore(state => state.stats);
	
	const colors = useStore.getState().colors;
	
	const [categories, setCategories] = useState(null);
	
	const setSelectedBook = useStore(state => state.setSelectedBook);
	
	useEffect(() => {
		if (!storedSubCats) return;
		const cats = createColorCategories(stats, colors, hoverShelf);
		setCategories(cats);
	}, [stats, hoverShelf, storedSubCats, colors]);
	
	useEffect(() => {
		setSide(Math.min(width, height) * 1.1);
	}, [width, height]);
	
	
	const handleFilterClick = (category) => {
		
		setSelectedBook(false)
		
		if (!category) {
			setFilterMembership(null);
			setHoveredColor(false);
			return
		}
		
		if (filterMembership?.subid) {
			setFilterMembership(filterMembership && category?.subid === filterMembership?.subid ? null : category);
			setHoveredColor(category);
		} else {
			setFilterMembership(filterMembership && category.id === filterMembership.id ? null : category);
			setHoveredColor(category);
		}
	};

	useEffect(() => {
		if (!filterMembership) {
			setFilterMembership(null);
			setHoveredColor(false);
		}
	}, [filterMembership, setFilterMembership, setHoveredColor]);
	
	if (categories == null) {
		return <p>Loading ...</p>;
	}
	
	return (
		<svg className={styles.svgDiv} width={'100%'} height={'100%'} ref={ref} style={{marginTop: '3em'}}
		     onMouseEnter={() => {setHoveredBook(false)}}>
			<g onClick={() => {
				setFilterMembership(null);
				setHoveredColor(false);
			}}>
				<rect
					x={0}
					y={0}
					width={'100%'}
					height={'100%'}
					fill={'#FFFFFF00'}
				/>
			</g>
			{categories.map((category, index) => (
				<g key={`category-${index}`}
				   style={{cursor: 'pointer'}}
				   transform={`translate(${side / 45 * Math.pow((category.data / 100), 0.5) * category.tX} ${side / 45 * Math.pow((category.data / 100), 0.5) * category.tY})`}
				   onClick={() => handleFilterClick(category)}
				   onMouseEnter={() => filterMembership ? null : setHoveredColor(category)}
				   onMouseLeave={() => filterMembership ? null : setHoveredColor(false)}>
					<circle
						r={side / 50 * Math.pow((category.data / 100), 0.5)}
						cx={width / 2} cy={height / 2}
						fill={hoverShelf == null ? category.fill : 'none'}
						stroke={hoverShelf != null ? category.fill : 'none'}
						strokeWidth={2}
						opacity={(filterMembership && filterMembership?.idx !== category?.idx)
						|| (hoveredColor && hoveredColor?.idx !== category?.idx)
							? 0.3 : 1}
					/>
					{hoverShelf != null && (
						<circle
							r={side / 50 * Math.pow((category.dataSub / 100), 0.5)}
							cx={width / 2} cy={height / 2}
							fill={category.fill}
							opacity={filterMembership && filterMembership?.idx !== category?.idx
								? 0.3 : 1}
						/>
					)}
					<text
						x={width / 2}
						y={height / 2 + category.textOffset}
						textAnchor='middle'
						fontSize={filterMembership && filterMembership?.idx === category?.idx ? side * 0.03 : side * 0.023}
						fontWeight={filterMembership && filterMembership?.idx === category?.idx ? 600 : 500}>
						<tspan alignmentBaseline='ideographic'>{category.label}</tspan>
					</text>
					<text
						x={width / 2}
						y={height / 2 + category.textOffset}
						textAnchor='middle'
						fontWeight={'bold'}
						fontSize={side * 0.018}>
						<tspan dy={isFirefox ? side * 0.03 : 0} alignmentBaseline='hanging'>
							{hoverShelf == null ? category.data : `${category.dataSub} / ${category.data}`}
						</tspan>
					</text>
				</g>
			))}
			{hoverShelf != null && (
				<text
					x={width / 2 + side * 0.025}
					y={height / 2 - side * 0.05}
					textAnchor='middle'
					fontSize={side * 0.02}
					fontWeight={'600'}
				>
					{hoverShelf.slice(0, -1)}
				</text>)}
		</svg>
	);
};
