import styles from "../styles.module.css";
import React, {useEffect, useRef, useState} from 'react';
import {useStore} from "../store";
import * as d3 from 'd3';

export function TaxonomyTag() {

	const hoveredTaxonomy = useStore(state => state.hoveredTaxonomy);
	const isGerman = useStore(state => state.isGerman);

	const renderTaxonomyPath = () => {
		if (!hoveredTaxonomy) return '';

		const path = hoveredTaxonomy.path || [];
		const currentName = isGerman ? hoveredTaxonomy.kurz : hoveredTaxonomy.brevia;

		if (hoveredTaxonomy.depth === 0) {
			return '';
		}

		return '| ' + path.map(item => isGerman ? item.kurz : item.brevia).join(' | ') +
			(path.length > 0 ? ' | ' : '') +
			currentName +
			` (${hoveredTaxonomy.value})`;
	};

	return (
		<div style={{
			backgroundColor: '#FFFFFF99',
		}}>
			<p>{renderTaxonomyPath()}</p>
		</div>
	);
}

export function BookTaxonomyTag() {
	const hoveredBook = useStore(state => state.hoveredBook);
	const isGerman = useStore(state => state.isGerman);
	const taxonomyd3 = useStore(state => state.taxonomyd3);

	const renderBookTaxonomyPath = () => {

		if (!hoveredBook || !taxonomyd3) return '';

		const wissensklasse = hoveredBook.Wissensklasse || '';
		const wissensunterklasse = hoveredBook.Wissensunterklasse || '';

		if (!wissensklasse && !wissensunterklasse) return '';

		let path = '';

		path += `${hoveredBook.Signatur || ''}: `;

		// Find the Wissensklasse in taxonomyd3
		const wissensklasseNode = taxonomyd3.children.find(child => child.name === wissensklasse);
		if (wissensklasseNode) {
			const wissensklasseText = isGerman ? wissensklasseNode.kurz : wissensklasseNode.brevia;
			path += `${wissensklasseText}`;

			// Find the Wissensunterklasse in the children of Wissensklasse
			const wissensunterklasseNode = wissensklasseNode.children.find(child => child.name === wissensunterklasse);
			if (wissensunterklasseNode) {
				const wissensunterklasseText = isGerman ? wissensunterklasseNode.kurz : wissensunterklasseNode.brevia;
				path += ` | ${wissensunterklasseText}`;
			}
		}

		return path;
	};

	return (
		<div style={{
			backgroundColor: renderBookTaxonomyPath() ? '#FFFFFF99' : '#FFFFFF00',
			padding: '5px',
			borderRadius: '3px',
			width: 'auto',
			height: 'auto'
		}}>
			<p style={{margin: 0, fontSize: '0.9em'}}>{renderBookTaxonomyPath()}</p>
		</div>
	);
}

export default function CirclePacking({inStory = false}) {
	
	const remInPx = useStore(state => state.remInPx);
	const taxonomyd3 = useStore(state => state.taxonomyd3);
	const dim = useStore(state => state.dim);
	const setFilterTaxonomy = useStore(state => state.setFilterTaxonomy);
	const filterTaxonomy = useStore(state => state.filterTaxonomy);
	const setHoveredTaxonomy = useStore(state => state.setHoveredTaxonomy);
	const hoveredTaxonomy = useStore(state => state.hoveredTaxonomy);
	const hoveredBook = useStore(state => state.hoveredBook);
	const isGerman = useStore(state => state.isGerman);
	const setIsGerman = useStore(state => state.setIsGerman);
	const mode = useStore(state => state.mode);
	const selectedLegend = useStore(state => state.selectedLegend);

	const svgRef = useRef(null);

	const [showNoDataOverlay, setShowNoDataOverlay] = useState(false);
	
	const showHighlights = useStore(state => state.showHighlights);
	

	// Function to get filtered data based on current mode and selection
	const getFilteredData = () => {
		if (!taxonomyd3 || !taxonomyd3.children) return taxonomyd3;

		let filteredData = {
			...taxonomyd3,
			children: taxonomyd3.children.map(node => ({
				...node,
				children: node.children.map(child => ({
					...child,
					// Use the appropriate pre-calculated value based on mode and selection
					value: selectedLegend ? (
						mode === 'memberships' ? (
							selectedLegend === 1 ? child.value_eugeniana :
								selectedLegend === 2 ? child.value_uncertain :
									selectedLegend === 3 ? child.value_noneugeniana : child.value
						) : mode === 'colors' ? (
							selectedLegend === 1 ? child.value_yellow :
								selectedLegend === 2 ? child.value_red :
									selectedLegend === 3 ? child.value_blue :
										selectedLegend === 4 ? child.value_uncertain :
											selectedLegend === 5 ? child.value_noneugeniana : child.value
						) : child.value
					) : child.value
				})).filter(child => child.value > 0)
			})).filter(node => node.children && node.children.length > 0)
		};

		return filteredData;
	};

	const createCirclePacking = (data) => {
		// console.log('Creating circle packing with data:', data, dim);

		if (!data || !data.children || data.children.length === 0) {
			// console.error("Invalid data structure for circle packing", data);
			return;
		}

		// Check if any children have values
		const hasValidChildren = data.children.some(child =>
			child.children && child.children.some(subChild => subChild.value > 0)
		);

		if (!hasValidChildren) {
			// console.error("No valid data points to display");
			return;
		}

		// Use different width/height calculations based on context
		const width = inStory ? (dim.screen.width * 0.4) : (dim.right.width * 0.95);
		const height = inStory ? (dim.screen.height * 0.85) : dim.right.height;

		const updateDelay = 1000;

		const svg = d3.select(svgRef.current)
			.attr('width', width + 3) // hotfix: outer circle is not trimmed anymore
			.attr('height', height)
			.attr('viewBox', [-width / 2, -height / 2, width, height])
			.attr('style', 'max-width: 100%; height: auto; font: 10px sans-serif;');

		const pack = data => d3.pack()
			.size([width, height])
			.padding(3)
			(d3.hierarchy(data)
				.sum(d => d.children ? 0 : (d.value || 1))
				.sort((a, b) => b.value - a.value));

		const root = pack(data);
		let focus = root;
		let view;

		const color = d3.scaleLinear()
			.domain([0, 5])
			.range(["hsl(100,5%,91%)", "hsl(100,10%,32%)"])
			.interpolate(d3.interpolateHcl);

		function calculateTotalValue(node) {
			if (!node.children) {
				return node.value || 0; // Return the node's value or 0 if it's not defined
			}
			return node.children.reduce((sum, child) => sum + calculateTotalValue(child), 0);
		}

		function setTaxonomyWithPath(d, setFunction) {
			if (d.depth === 0) {
				if (setFunction === setFilterTaxonomy) {
					setFunction(null); // Set filter to null for root level
				} else {
					setFunction({
						name: d.data.name,
						depth: d.depth,
						german: d.data.german,
						kurz: d.data.kurz,
						brevia: d.data.brevia,
						value: calculateTotalValue(d),
						path: []
					});
				}
			} else {
				const path = [];
				let current = d;
				while (current.parent && current.depth > 1) {
					path.unshift({
						name: current.parent.data.name,
						kurz: current.parent.data.kurz,
						brevia: current.parent.data.brevia
					});
					current = current.parent;
				}

				setFunction({
					name: d.data.name,
					depth: d.depth,
					german: d.data.german,
					kurz: d.data.kurz,
					brevia: d.data.brevia,
					value: calculateTotalValue(d),
					path: path
				});
			}
		}

		// Set initial taxonomy state
		setFilterTaxonomy(null); // Set initial filter to null
		setTaxonomyWithPath(root, setHoveredTaxonomy);

		let hoverTimer;
		let selectedNode = null;

		const node = svg.append("g")
			.selectAll("circle")
			.data(root.descendants())
			.join("circle")
			.attr("fill", d => d.children ? color(d.depth) : "white")
			.attr("pointer-events", "all")
			.on("mouseover", function (event, d) {
				event.stopPropagation();
				d3.select(this).attr("stroke", "purple").attr("stroke-width", 2);

				clearTimeout(hoverTimer);

				hoverTimer = setTimeout(() => {
					setTaxonomyWithPath(d, setHoveredTaxonomy);
				}, updateDelay);
			})
			.on("mouseout", function (event) {
				event.stopPropagation();
				d3.select(this).attr("stroke", null).attr("stroke-width", null);

				clearTimeout(hoverTimer);
				setTaxonomyWithPath(root, setHoveredTaxonomy);
			})
			.on("click", (event, d) => {
				event.stopPropagation();
				clearTimeout(hoverTimer);
				if (focus !== d) {
					selectedNode = d;
					zoom(d);
					event.stopPropagation();
				}
			});

		function updateNodeColors() {
			node.attr("fill", d => {
				if (d === selectedNode) {
					if (d.depth === 0) {
						return "#f6e7ee"; // Light purple for selected node
					} else if (d.depth === 1) {
						return "#E2C0CD"; // Light purple for selected node
					} else if (d.depth === 2) {
						return "#FFF0F9"; // Light purple for selected node
					}
				} else if (d.children) {
					return color(d.depth);
				} else {
					return "white";
				}
			});
		}

		const label = svg.append("g")
			.style("font-family", "Abel, sans-serif")
			.attr("pointer-events", "none")
			.attr("text-anchor", "middle")
			.selectAll("text")
			.data(root.descendants())
			.join("text")
			.style("fill-opacity", d => d.parent === root ? 1 : 0)
			.style("display", d => d.parent === root ? "inline" : "none")
			.style("fill", "black")
			.style("font-size", d => d.depth === 1 ? "1.7em" : "1.2em")
			.style("stroke", "white")
			.style("stroke-width", "0.5px")
			.style("paint-order", "stroke fill")
			.style("text-shadow", "0 0 3px white, 0 0 3px white, 0 0 3px white, 0 0 3px white"); // Add white shadow


		// Improved function to split text into two lines
		function splitText(text = "", maxLength = 12) {
			if (!text || typeof text !== 'string') return [""];
			if (text.length <= maxLength) return [text];

			// Find the last space before or at maxLength
			let splitIndex = text.lastIndexOf(' ', maxLength);

			// If no space found, look for the next space after maxLength
			if (splitIndex === -1) {
				splitIndex = text.indexOf(' ', maxLength);
			}

			// If still no space found, split at maxLength
			if (splitIndex === -1) {
				splitIndex = 3 * maxLength;
			}

			// Ensure we don't split too close to the end
			if (text.length - splitIndex < 4 && splitIndex !== maxLength) {
				return [text];
			}

			return [
				text.slice(0, splitIndex).trim(),
				text.slice(splitIndex).trim()
			].filter(Boolean); // Remove empty strings
		}

		// Add tspan elements for each line of text
		label.each(function (d) {
			const text = isGerman ? d.data.kurz : d.data.brevia;
			const lines = splitText(text);
			const textElement = d3.select(this);
			textElement.selectAll("*").remove(); // Clear existing content
			lines.forEach((line, i) => {
				textElement.append("tspan")
					.attr("x", 0)
					.attr("dy", i === 0 ? "-0.2em" : "1.2em")
					.text(line || ""); // Use empty string if line is undefined
			});
		});

		svg.on("click", () => {
			selectedNode = null;
			zoom(root);
		});

		zoomTo([root.x, root.y, root.r * 2]);

		function zoomTo(v) {
			const k = width / v[2];

			view = v;

			// Use D3's selection.attr() for efficient batch updates
			node.attr("transform", d => `translate(${(d.x - v[0]) * k},${(d.y - v[1]) * k})`)
				.attr("r", d => d.r * k);

			label.attr("transform", d => `translate(${(d.x - v[0]) * k},${(d.y - v[1]) * k})`)
				.style("font-size", d => {
					const baseSize = d.depth === 1 ? 1.3 : 1;
					return `${baseSize * Math.min(k, 1)}em`;
				});
		}

		function zoom(d) {
			const focus0 = focus;
			focus = d;

			// Disable hover events during zoom
			node.on("mouseover", null).on("mouseout", null);
			node.on("mouseleave", null).on("mouseenter", null);

			label.style("text-shadow", "none");

			const transition = svg.transition()
				.duration(1050)
				.tween("zoom", d => {
					const i = d3.interpolateZoom(view, [focus.x, focus.y, focus.r * 2]);
					return t => {
						zoomTo(i(t));
					};
				});

			// Update visibility and opacity during transition
			label
				.transition(transition)
				.style("fill-opacity", d => {
					if (focus === root) return d.depth === 1 ? 1 : 0;
					if (focus.depth === 1) return d.depth <= 2 ? 1 : 0;
					return d.depth === focus.depth ? 1 : 0;
				})
				.style("display", d => {
					if (focus === root) return d.depth === 1 ? "inline" : "none";
					if (focus.depth === 1) return d.depth <= 2 ? "inline" : "none";
					return d.depth === focus.depth ? "inline" : "none";
				});

			// After transition, update font sizes and positions
			transition.end().then(() => {

				label.each(function (d) {
					const textElement = d3.select(this);
					let fontSize;
					if (focus === root) {
						fontSize = d.depth === 1 ? 1.4 : 1.2;
					} else if (focus.depth === 1) {
						fontSize = d.depth === 1 ? 1.8 : 1.4;
					} else {
						fontSize = 1.6;
					}
					textElement.style("font-size", `${fontSize}em`)
						.style("text-shadow", `0 0 ${fontSize * 2}px white, 0 0 ${fontSize * 2}px white, 0 0 ${fontSize * 2}px white, 0 0 ${fontSize * 2}px white`); // Adjust shadow size based on font size

					const tspans = textElement.selectAll("tspan");
					const lineCount = tspans.size();
					let baseOffset = focus.depth === 1 && d.depth === 1 ? -1.4 : 0;

					tspans.attr("dy", (_, i) => {
						if (lineCount === 1) return `${baseOffset + 0.35}em`;
						return i === 0 ? `${baseOffset - 0.5 * (lineCount - 1)}em` : "1.2em";

					});
				});

				updateNodeColors();

				if (d.depth === 0) {
					setFilterTaxonomy(null);
				} else {
					setTaxonomyWithPath(d, setFilterTaxonomy);
				}

				setTaxonomyWithPath(d, setHoveredTaxonomy);

				// Re-enable hover events
				node.on("mouseover", function (event, d) {
					event.stopPropagation();
					d3.selectAll("circle").attr("stroke", null).attr("stroke-width", null);
					d3.select(this).attr("stroke", "purple").attr("stroke-width", 2);

					clearTimeout(hoverTimer);
					hoverTimer = setTimeout(() => {
						setTaxonomyWithPath(d, setHoveredTaxonomy);
					}, updateDelay);
				})
					.on("mouseout", function (event) {
						event.stopPropagation();
						d3.selectAll("circle").attr("stroke", null).attr("stroke-width", null);
						clearTimeout(hoverTimer);
						setTaxonomyWithPath(focus, setHoveredTaxonomy);
					});
			});
		}

	}

	useEffect(() => {
		if (!filterTaxonomy) {
			d3.select(svgRef.current).selectAll("*").remove();
            if ((!showHighlights || inStory) && dim.screen.width > 0 ) {  // Only create circle packing if not showing highlights
			createCirclePacking(getFilteredData());
		}
        }
    }, [filterTaxonomy, showHighlights, dim]);

    // Effect for handling dimension and other changes
	useEffect(() => {
        if (taxonomyd3 && (!showHighlights || inStory)) {  // Only proceed if not showing highlights
			const filteredData = getFilteredData();
			if (filteredData && filteredData.children && filteredData.children.length > 0) {

				setShowNoDataOverlay(false);
				d3.select(svgRef.current).selectAll("*").remove();
				if (dim.screen.width > 0) {
					createCirclePacking(filteredData);
				}
			} else {
				// console.log('No valid data to display');
				d3.select(svgRef.current).selectAll("*").remove();
				setShowNoDataOverlay(true);
			}
		}
    }, [taxonomyd3, dim, isGerman, mode, selectedLegend, showHighlights]);

	// Effect for handling hoveredBook changes
	useEffect(() => {
        if (!svgRef.current || (showHighlights && !inStory)) return;  // Add showHighlights check

		const svg = d3.select(svgRef.current);
		const circles = svg.selectAll("circle");

        if (hoveredBook) {
			circles.each(function (d) {
                if (d.data.name === hoveredBook.Wissensunterklasse) {
					d3.select(this).attr("stroke", "purple").attr("stroke-width", 2);
				} else {
					d3.select(this).attr("stroke", null).attr("stroke-width", null);
				}
			});
		} else {
			// Remove highlight from all circles when no book is hovered
			circles.attr("stroke", null).attr("stroke-width", null);
		}

		// Cleanup function to remove highlights when component unmounts or before next effect run
		return () => {
			circles.attr("stroke", null).attr("stroke-width", null);
		};
	}, [hoveredBook, showHighlights]);

    // Now return null if showing highlights
    if (showHighlights && !inStory) {
        return null;
    }

	
	return (
    <div style={{ position: 'relative' }}>
			{!inStory && (
				<div style={{
					width: 'auto',
					position: 'fixed',
					top: dim.top.height +remInPx*1.25,
					left: dim.left.width + dim.middle.width + remInPx*8,
					marginRight: '40px',
					pointerEvents: 'none',
				}}>
					<TaxonomyTag/>
				</div>
			)}
			{!inStory && (
				<div style={{
					width: 'auto',
					position: 'fixed',
					top: dim.top.height +remInPx*3.5,  // Positioned below TaxonomyTag
					left: dim.left.width + dim.middle.width + remInPx*8,
					marginRight: '40px',
					pointerEvents: 'none',
				}}>
					<BookTaxonomyTag/>
				</div>
			)}
			{!inStory && (
				<div style={{position: 'fixed', top: dim.top.height + remInPx*3, right: remInPx*2}}>
					<div
						className={styles.Button}
						style={{width: 'max-content'}}
						onClick={() => setIsGerman(!isGerman)}
					>
						{isGerman ? 'Latein' : 'Deutsch'}
					</div>
				</div>
			)}
			<div className={styles.CirclePackingContainer}>
				<svg ref={svgRef}></svg>
        {showNoDataOverlay && (
          <NoDataOverlay
            inStory={inStory}
          />
        )}
			</div>
		</div>
	);
}

const NoDataOverlay = ({ inStory }) => {
	return (
		<div style={{
			position: 'absolute',
			top: 0,
			left: 0,
			right: 0,
			bottom: 0,
			display: 'flex',
			flexDirection: 'column',
			alignItems: 'center',
			justifyContent: 'center',
			padding: '2rem',
			marginBottom: '6rem',
			gap: '1.5rem',
			borderRadius: inStory ? '8px' : '0',
			zIndex: 10
		}}>
			<div style={{
				textAlign: 'center',
				maxWidth: '400px'
			}}>
				<h3 style={{ marginBottom: '1rem', color: '#333' }}>
					Für die Auswahl "Nicht Eugeniana" sind keine Daten vorhanden.
				</h3>
				<p style={{ color: '#666', lineHeight: '1.5' }}>
					Da nur Bücher der Eugeniana nach Leibnitz klassifiziert wurden, wählen Sie bitte eine andere Kategorie aus. Sie können auch die Filterung aufheben, indem Sie noch einmal auf 'Nicht Eugeniana' klicken.
				</p>
			</div>
		</div>
	);
};
