/* eslint-disable react/prop-types */
import styles from "../styles.module.css";
import React, {useRef, useState, useEffect, memo} from 'react';
import {useStore} from "../store";
import {List} from 'react-virtualized';
import {useThrottle} from "../lib/hooks.jsx";
import {VscFilter} from "react-icons/vsc";
import {SearchFilter} from "./Search.jsx";
import * as d3 from "d3";

export const BookList = (props) => {

	const {height, width} = props;

	const searchContainer = useRef()
	const refHead = useRef()

	const [headHeight, setHeadHeight] = useState(30)
	const [searchHeight, setSearchHeight] = useState()

	const maxRows = 4000;

	const currentShelf = useStore(state => state.currentShelf);
	const storedSubCats = useStore(state => state.storedSubCats);
	const storedCats = useStore(state => state.storedCats);

	const chartType = useStore(state => state.chartType);

	const mode = useStore(state => state.mode);
	const dim = useStore(state => state.dim);

	const selectedBook = useStore(state => state.selectedBook)
	const setSelectedBook = useStore(state => state.setSelectedBook)

	const search = useStore(state => state.search);
	const setSearch = useStore(state => state.setSearch);

	const setFilterMembership = useStore(state => state.setFilterMembership);
	const filterMembership = useStore(state => state.filterMembership);
	const filterTimerange = useStore(state => state.filterTimerange);
	const setFilterTimerange = useStore(state => state.setFilterTimerange)
	const filterPlace = useStore(state => state.filterPlace);
	const setFilterPlace = useStore(state => state.setFilterPlace);
	const filterLanguage = useStore(state => state.filterLanguage);
	const setFilterLanguage = useStore(state => state.setFilterLanguage);
	const filterTaxonomy = useStore(state => state.filterTaxonomy);
	const setFilterTaxonomy = useStore(state => state.setFilterTaxonomy);
	const isGerman = useStore(state => state.isGerman);
	const translateLanguage = useStore(state => state.translateLanguage);
	const selectedLegend = useStore(state => state.selectedLegend);
	const setSelectedLegend = useStore(state => state.setSelectedLegend);

	const [limit, setLimit] = useState(maxRows)
	const [showMore, setShowMore] = useState(false)
	const [displayData, setDisplayData] = useState()

	const [rowHeight, setRowHeight] = useState(100)

	useEffect(() => {
		setSelectedBook(false)
	}, [currentShelf, setSelectedBook]);

	useEffect(() => {
		//console.log(filterMembership)
		if (!filterMembership) {
			setSelectedLegend(false)
		}
	}, [filterMembership]);

	useEffect(() => {
		//console.log('filterplace changed', filterPlace)
	}, [filterPlace]);

	useEffect(() => {
		// console.log('filterlanguage changed', filterLanguage)
	}, [filterLanguage]);

	useEffect(() => {
		// console.log('filtertaxonomy changed', filterTaxonomy)
	}, [filterTaxonomy]);

	useEffect(() => {
		if (!storedSubCats) return
		setDisplayData(storedSubCats.all[currentShelf])
		setShowMore(false)
	}, [currentShelf, storedSubCats])

	useEffect(() => {
		setLimit(showMore ? displayData.length : maxRows)
	}, [showMore]);


	const sortBy = (dataset, attribute = 'date', direction = 'asc') => {

		if (attribute === 'date') {
			const now = Date.now();
			const result = dataset.sort((a, b) => {
				const sortYearA = !isNaN(new Date(a['Anfang Veröffentlichungsdatum'])) ? new Date(`${a['Anfang Veröffentlichungsdatum']}`) : now;
				const sortYearB = !isNaN(new Date(b['Anfang Veröffentlichungsdatum'])) ? new Date(`${b['Anfang Veröffentlichungsdatum']}`) : now;
				return direction === 'asc' ? sortYearA - sortYearB : sortYearB - sortYearA;
			});
			return result
		} else if (attribute === 'signatur') {
			const result = dataset.sort((a, b) => a['signaturIndex'] - b['signaturIndex'])
			return result

		} else if (attribute === 'Autor') {

			const sortable = dataset.filter(d => d['Autor'])
			const notsortable = dataset.filter(d => !d['Autor'])

			const sorted = sortable.sort((a, b) => {
				return direction === 'asc' ? a['Autor'].localeCompare(b['Autor']) : b['Autor'].localeCompare(a['Autor'])
			});

			const result = [...sorted, ...notsortable]

			return result
		}

	}
	
	useEffect(() => {

		if (!storedCats) return

		var newData;

		if (!currentShelf && !filterMembership) {
			let all = [...storedCats.eugeniana, ...storedCats.noneugeniana, ...storedCats.uncertain]
			newData = !filterMembership ? sortBy(all, 'signatur', 'asc') : all.filter((d) => filterMembership.value === d.membership)

		} else if (!currentShelf && filterMembership) {
			let all = [...storedCats.eugeniana, ...storedCats.noneugeniana, ...storedCats.uncertain]
			if (filterMembership.subid) {
				newData = !filterMembership ? sortBy(all, 'signatur', 'asc') : all.filter((d) => filterMembership.value === d.membership && filterMembership.subid === d.Farbklassifizierung)
			} else {
				newData = !filterMembership ? sortBy(all, 'signatur', 'asc') : all.filter((d) => filterMembership.value === d.membership)
			}
		} else if (!filterMembership && storedSubCats && mode === "overview") {
			newData = storedSubCats.all[currentShelf]
		} else if (storedSubCats) {
			if (filterMembership?.subid) {
				newData = storedSubCats.all[currentShelf]
					.filter((d) => filterMembership.value === d.membership && filterMembership.subid === d.Farbklassifizierung)
			} else if (filterMembership) {
				newData = storedSubCats.all[currentShelf].filter((d) => filterMembership.value === d.membership)
			} else {
				newData = storedSubCats.all[currentShelf].filter((d) => d)
			}
		}

		if (filterPlace) {
			const filtered = newData.filter(book => {
				if (book['Orte_corr'].includes(';')) {
					let splitted = book['Orte_corr'].split(';')
					let match = false
					splitted.map(d => {
							if (d === filterPlace && !d.membership) match = true
						}
					)
					return match
				}
				else { return book['Orte_corr'] === filterPlace}

			})
			newData = filtered
		}

		if (filterLanguage) {
			const filtered = newData.filter(book => {
				return book['Sprache'] === filterLanguage ? true : false
			})
			newData = filtered
		}

		if (filterTaxonomy) {
			const filtered = newData.filter(book => {
				if (filterTaxonomy.depth === 1) {
					return book['Wissensklasse'] === filterTaxonomy.name ? true : false
				}
				else if (filterTaxonomy.depth === 2) {
					return book['Wissensunterklasse'] === filterTaxonomy.name ? true : false
				}
				else return false
			})
			newData = filtered
		}

		if (search !== '') {
			const filtered = newData.filter(book => {
					let Search = search.toUpperCase()
					return book['Autor']?.toUpperCase().includes(Search) ||
						book['Signatur']?.toUpperCase().includes(Search) ||
						book['Titel']?.toUpperCase().includes(Search) ||
						book['Orte_corr']?.toUpperCase().includes(Search) ||
						book['Sprache']?.toUpperCase().includes(Search) ||
						book['Barcode']?.toUpperCase().includes(search) ||
						book['Anfang Veröffentlichungsdatum'].includes(Search)
				}
			)

			if (filterPlace) {
				const filtered = newData.filter(book => book['Orte_corr'].includes(filterPlace) ? true : false)
				newData = filtered
			}

			setDisplayData(filtered)

		} else {
			setDisplayData(newData)
		}


	}, [filterMembership, currentShelf, search, filterTimerange, chartType, filterPlace, filterLanguage, filterTaxonomy, storedCats]);


	const filterByTimerange = (dataset) => {
		const filteredData = dataset.filter(
			d => d["Anfang Veröffentlichungsdatum"] >= filterTimerange?.x0 && d["Anfang Veröffentlichungsdatum"] <= filterTimerange?.x1
		)
		return filteredData
	}

	useEffect(() => {

		if (refHead.current) {
			setHeadHeight(refHead.current.clientHeight)
		}
		if (searchContainer.current) {
			setSearchHeight(searchContainer.current.clientHeight)
		}
	}, [height, width, chartType, mode, filterTimerange, search, displayData]);

	useEffect(() => {
		if (!displayData) return
	}, [displayData]);

	useEffect(() => {
		// get height of bookItem by measuring a invisible dummy bookItem
		let sel = d3.select('.dummyItem')
		if (sel && sel.selectChild().node()) {
			setRowHeight(sel.selectChild().node().getBoundingClientRect().height)
		}
	}, [dim])

	return (
		<>
			{displayData &&
				<div height={'100%'} className={styles.bookListContainer}>
					<div className={styles.bookItemHead} ref={refHead}>
						{(mode.includes("colors") || mode.includes("memberships")) && filterTimerange && displayData ?
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? '1 / ' : ''}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}" </span>}
								{filterByTimerange(displayData).length} Bücher{selectedBook ? 'n' : ''}
								<br/> von {Math.round(filterTimerange.x0)} - {Math.round(filterTimerange.x1)}
							</h4>
							: (mode.includes("colors") || mode.includes("memberships")) && filterPlace && displayData ?
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? ' 1 / ' + displayData.length + ' Büchern' : displayData.length + ' Bücher'}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}"</span>}
								<br/> <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setFilterPlace(false)}/> aus {filterPlace}
							</h4>
							: (mode.includes("colors") || mode.includes("memberships")) && filterLanguage && displayData ?
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? ' 1 / ' + displayData.length + ' Büchern' : displayData.length + ' Bücher'}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}"</span>}
								<br/> <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setFilterLanguage(false)}/> {translateLanguage(filterLanguage)}
							</h4>
							: (mode.includes("colors") || mode.includes("memberships")) && filterMembership && displayData ?
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? ' 1 / ' + displayData.length + ' Büchern' : displayData.length + ' Bücher'}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}"</span>}
								<VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setFilterMembership(null)}/>
							</h4>
							: (mode.includes("colors") || mode.includes("memberships")) && filterTaxonomy && displayData ?
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? ' 1 / ' + displayData.length + ' Büchern' : displayData.length + ' Bücher'}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}"</span>}
								<VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setFilterTaxonomy(null)}/> in {isGerman ? filterTaxonomy.kurz : filterTaxonomy.brevia}
							</h4>
							:
							<h4 onClick={() => {setSelectedBook(false)}}>
								{currentShelf ? currentShelf.slice(0, -1) : 'Alle'}
								&nbsp;&mdash;&nbsp;
								{selectedBook && <VscFilter className={styles.reactIcons} size={'1.3em'}/>}
								{selectedBook ? ' 1 / ' + displayData.length + ' Büchern' : displayData.length + ' Bücher'}
								{search !== '' && <VscFilter className={styles.reactIcons} size={'1.3em'} onClick={()=>setSearch('')}/>}
								{search !== '' && <span> "{search}"</span>}
							</h4>


						}
						<div className={styles.bookItemHeadTag} onClick={()=>setFilterMembership(null)}
						     style={{
						        background: filterMembership ? filterMembership.fill : '#ffffff00',
						        border: filterMembership ? '1px solid #aaaaaa' : '0px solid #ffffff00'
						        }}
						     />
					</div>

					<BookListRows className={styles.bookList}
						data={(mode.includes('colors') || mode.includes('memberships')) && chartType === "time" ?
							filterByTimerange(displayData)
							:
							displayData}
						limit={limit}
						height={searchHeight && headHeight ? height - headHeight - searchHeight : height - 280}
						width={width}
						rowHeight={rowHeight}
					/>
					{1 == 2 && limit < displayData.length &&
						<div onClick={() => setShowMore(!showMore)} className={styles.bookShowMore}>
							{showMore ? 'Weniger anzeigen' : 'Alle anzeigen'}
						</div>
					}

					<div className={styles.searchContainerBooklist} ref={searchContainer}>
						<SearchFilter/>
					</div>

					<div className={"dummyItem"} style={{position: 'fixed', left:200, top: 200, width: 100, visibility: 'hidden',}}>
						<BookItem key={'fakerow'} book={displayData[0]}/>
					</div>

				</div>
			}
		</>
	)
}

const BookListRows = React.memo((props) => {

		const listRef = useRef(null);

		const {data, height, width, rowHeight} = props;

		const shelvesList = useStore.getState().shelvesList;
		const sections = useStore.getState().ZZsections;
		const storedSubCats = useStore.getState().storedSubCats;

		const hoveredBook = useStore(state => state.hoveredBook)

		const selectedBook = useStore(state => state.selectedBook)
		const setSelectedBook = useStore(state => state.setSelectedBook)
		const currentShelf = useStore(state => state.currentShelf)
		const zoomShelf = useStore(state => state.zoomShelf)

		const [showSelectedBook, setShowSelectedBook] = useState(false)
	
		useEffect(() => {
			if (listRef.current) {
				listRef.current.recomputeRowHeights();
			}
		}, [rowHeight])

		useEffect(() => {
			if (!zoomShelf) setSelectedBook(false)
		}, [zoomShelf, setSelectedBook]);


		useEffect(() => {
			setShowSelectedBook(currentShelf && selectedBook ? selectedBook : false)
		}, [selectedBook, currentShelf]);

		useEffect(() => {
			if (listRef.current && hoveredBook) {
				listRef.current.scrollToRow(data.findIndex(b => b === hoveredBook));
				setTimeout(() => {
					try {
						listRef.current.forceUpdateGrid();
					} catch {
					}
				}, [1500])
			}

			// get height of bookItem by measuring a invisible dummy bookItem

			if (hoveredBook) {
			
				if (!hoveredBook['Signatur_corr']?.startsWith('BE')) return;

				let signatur = hoveredBook['Signatur_corr']

				let s1 = shelvesList.findIndex(f => signatur.startsWith(f + '.'))
				let s2 = sections.findIndex(f => signatur.includes('.' + f + '.'))

				let templist = storedSubCats.all[shelvesList[s1] + '.']?.filter(f => f['Signatur_corr'].includes('.' + sections[s2]))
				let s3 = templist?.findIndex(f => f['Signatur_corr'] === signatur)

				let sel = d3.select('#book-' + s1 + '-' + s2 + '-' + s3)

				var x, y, h;
				if (!sel.empty()) {
					x = sel.attr('x')
					y = sel.attr('y')
					h = sel.attr('height')
				}

				var circle = d3.selectAll("circle[id*='markercircle']")
				if (!circle.empty()) circle.attr('opacity', 0)

				d3.select('#markercircle-' + s1 + '-' + s2)
					.attr('cx', x)
					.attr('cy', parseFloat(y) + parseFloat(h) / 2)
					.attr('opacity', 1);

			} else {
				d3.selectAll("circle[id*='markercircle']").attr('opacity', 0)
			}

		}, [hoveredBook]);

		function renderRow({index, key, style}) {
			return (
				<div key={key} style={style} className={styles.bookRow}>
					<BookItem key={`index-${index}`} book={showSelectedBook ? showSelectedBook : data[index]}/>
				</div>
			);
		}

		return (
			<div className={styles.bookList} width={'100%'}>
				{data.length > 0 &&
					<List
						ref={listRef}
						width={width}
						height={height}
						rowHeight={rowHeight}
						rowRenderer={renderRow}
						rowCount={showSelectedBook ? 1 : data.length}
						overscanRowCount={3}
					/>
				}
			</div>
		)
	}
)
BookListRows.displayName = 'BookListRows';

const BookItem = memo((props) => {

	const {book} = props;

	const dim = useStore(state => state.dim);
	const throttledValue = useThrottle(dim, 500);
	const [restarting, setRestarting] = useState(false)

	const showHintNonDigitized = useStore(state => state.showHintNonDigitized)
	const setShowHintNonDigitized = useStore(state => state.setShowHintNonDigitized)

	const showCloseup = useStore(state => state.showCloseup);
	const setShowCloseup = useStore(state => state.setShowCloseup);

	const hoveredBook = useStore(state => state.hoveredBook);
	const setHoveredBook = useStore(state => state.setHoveredBook);

	const mode = useStore(state => state.mode);

	const [hovered, setHovered] = useState(false)

	const colors = useStore(state => state.colors)

	const handleClick = (e) => {

		if (!e || book['IIIF Manifest'] === "") {
			setShowHintNonDigitized(!showHintNonDigitized)
			return
		}

		if (showCloseup !== e) {
			setShowCloseup(false)
			setTimeout(() => {
				setShowCloseup(e)
			}, 50)
		} else {
			setShowCloseup(false)
		}
	}

	useEffect(() => {
		if (showCloseup && !restarting && showCloseup === book["Barcode"]) {
			const oldvalue = showCloseup;
			setShowCloseup(false)
			if (!restarting) {
				setTimeout(() => {
					setShowCloseup(oldvalue)
					setRestarting(false)
				}, 1500)
				setRestarting(true)
			}
		}
	}, [throttledValue]);

	const getBookColors = (color, membership) => {
		if (membership === 1) {
			return colors.Farbklassifizierung[color]
		} else {
			return membership === 2 ? colors.Farbklassifizierung['uncertain'] : colors.Farbklassifizierung['noneugeniana']
		}
	}

	// lines in book icon
	const getBookInnerColors = (color, membership) => {
		if (membership === 1) {
			return colors.FarbklassifizierungInner[color]
		} else {
			return membership === 2 ? colors.FarbklassifizierungInner['uncertain'] : colors.membershipInner[membership]
		}
	}

	useEffect(() => {}, [hoveredBook])

	if (!book) return <></>

	return (
		<div className={styles.bookItem}
		     style={{
			     background: book && showCloseup === book["Barcode"] ? '#FFFFFF' : (hovered || hoveredBook === book) ? '#dfdfdf99' : 'inherit',
			     fontStyle: hovered ? 'italic' : 'normal',
			     cursor: book["Barcode"] ? "pointer" : "default",
			     userSelect: 'none',
		     }}
		     onClick={() => handleClick(book["Barcode"])}
		     onMouseOver={() => {
			     setHovered(book["Barcode"]);
			     setHoveredBook(book)
		     }}
		     onMouseLeave={() => {
			     setHovered(false);
			     setHoveredBook(false)
		     }}
		>
				{
					book && book["Barcode"] != ''  && book["IIIF Manifest"] !== '' ?
//					book && book["Barcode"] != '' ?
						<div className={styles.bookItemIcon} onClick={() => handleClick(book["Barcode"])}>
							<BookIcon fill=
								          {mode === "colors" ?
									          getBookColors(book['Farbklassifizierung'], book['membership'])
									          :
									          colors['membership'][book['membership']]
								          }
							          strokeWidth={1}
							          innerStroke=
								          { mode === "colors" ?
									          getBookInnerColors(book['Farbklassifizierung'], book['membership'])
									          :
								              colors['membershipInner'][book['membership']]
								          }
							/>
						</div>
						:
						<div className={styles.bookItemIcon}>
						<BookIcon fill={
							mode === "placement" || mode === "colors" ?
								getBookColors(book['Farbklassifizierung'], book['membership'])
								:
								colors['membership'][book['membership']]
						} stroke='grey' strokeWidth={1}
						          innerStroke={
										'#FFFFFF00'
						          }
						/>
						</div>
				}

			<div className={styles.bookItemText}
			>
				<p className={styles.bookMetadata}><BookMetadata book={book}/></p>
				<p className={styles.bookTitle}>{book['Titel']}</p>
				<p className={styles.bookAuthor}>{book['Autor'] != "" ? book['Autor'] : "ohne Autor"}</p>
			</div>

		</div>
	)
})
BookItem.displayName = 'BookItem';

export const BookMetadata = (props) => {

	const {book} = props;
	const metadata = []
	const date = (book['Anfang Veröffentlichungsdatum'] != "" && book['Ende Veröffentlichungsdatum'] != "") ? `${book['Anfang Veröffentlichungsdatum']} - ${book['Ende Veröffentlichungsdatum']}` : book['Anfang Veröffentlichungsdatum'] != "" ? book['Anfang Veröffentlichungsdatum'] : "kein Datum";

	if (date) {
		metadata.push(date);
	}

	metadata.push(book["Orte_corr"]?.replace(';', ', '));

	return <span>{metadata.join(" | ")}</span>
}

export const BookIcon = (props) => {

	const {fill, stroke, innerStroke} = props;

	const fillColor = fill ? fill : 'lightgrey';
	const strokeColor = stroke ? stroke : '#213547';
	const innerStrokeColor = innerStroke ? innerStroke : '#213547';

	return (
		<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg"
		     width="24"
		     height="24"
		     viewBox="0 0 24 24"
		     strokeWidth="1"
		     strokeLinecap="round"
		     strokeLinejoin="round"
		     stroke={strokeColor}
		>
			<g fill={fillColor}>
				<path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/>
				<path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/>
			</g>
			<g stroke={innerStrokeColor}>
				<path d="M6 8h2"/>
				<path d="M6 12h2"/>
				<path d="M16 8h2"/>
				<path d="M16 12h2"/>
			</g>
		</svg>
	)
}
