/* eslint-disable react/prop-types */
import styles from "./styles.module.css";

import React, {useEffect, useState, useRef, useCallback} from 'react';
import {useStore} from "./store";
import {useMeasure} from "react-use";
import LinesEllipsis from 'react-lines-ellipsis'
import PropTypes from 'prop-types';
import * as d3 from 'd3';

import {logVersion} from './lib/version.jsx'
import {BookIcon, BookList} from './components/BookList'
import {OvalSvg} from './components/OvalSvg'
import {MembershipView} from './components/MembershipView'
import {Preprocess} from "./components/Preprocess";
import {NavBar} from "./components/NavBar.jsx";
import {CombinedViewer} from "./components/Mirador.jsx";
import {useThrottle} from "./lib/hooks.jsx";
import {ColorView} from "./components/ColorView.jsx";
import {TimeLine} from "./components/Timeline.jsx";
import CirclePacking from "./components/CirclePacking.jsx";
import {Places} from "./components/Places.jsx";
import {Screen} from "./components/Screen";
import {Tutorial} from "./components/Tutorial.jsx";
import {Hint} from "./components/Hints.jsx";
import {Intro} from "./components/Intro.jsx";
import {GeoMap} from "./components/Map.jsx";
import {Languages} from "./components/Languages.jsx";
import queryString from 'query-string';
import {Highlights} from "./components/Highlights.jsx";
import {Stories} from "./components/Story.jsx";
import {Onboarding} from "./components/Onboarding.jsx";
import {Legend} from "./components/Legend.jsx";

export const AppWrapper = () => {
	return (
		<>
			<Intro />
			<App />
		</>
	);
};

function App() {
	
	const store = useStore();
	
	const [refTop, top] = useMeasure();
	const [refLeft, left] = useMeasure();
	const [refMiddle, middle] = useMeasure();
	const [refRight, right] = useMeasure();
	const [refScreen, screen] = useMeasure();
	
	const [dimchanged, setDimchanged] = useState(false);
	const throttledValue = useThrottle(dimchanged, 500);
	
	const [isBarcode, setIsBarcode] = useState(false);
	
	const setPlaces = useStore(state => state.setPlaces);
	
	const firstRender = useStore(state => state.firstRender);
	const setFirstRender = useStore(state => state.setFirstRender);
	
	const showWelcome = useStore(state => state.showWelcome);
	const showHighlights = useStore(state => state.showHighlights);
	
	const showStories = useStore(state => state.showStories);
	
	const setShowIntro = useStore(state => state.setShowIntro);
	const setShowHighlights = useStore(state => state.setShowHighlights);
	const setChartType = useStore(state => state.setChartType);
	
	const showCloseup = useStore(state => state.showCloseup);
	const mode = useStore(state => state.mode);
	const chartType = useStore(state => state.chartType);
	const legend = useStore(state => state.legend);
	
	const dim = useStore(state => state.dim);
	const setDim = useStore(state => state.setDim);
	const setMouseScreen = useStore(state => state.setMouseScreen);
	
	const hoveredBook = useStore(state => state.hoveredBook);
	const setHoveredBook = useStore(state => state.setHoveredBook);
	
	const setWasTouched = useStore(state => state.setWasTouched)
	
	const search = useStore(state => state.search)
	const setSearch = useStore(state => state.setSearch)
	
	const setRemInPx = useStore(state => state.setRemInPx);
	
	const titleChart = {
		"memberships": {
			"time": "Zeitlicher Ursprung des Bestands",
			"geo": "Räumlicher Ursprung des Bestands",
			"language": "Sprachen",
			"geomap": "Räumlicher Ursprung des Bestands",
			"member": "Zugehörigkeit zur Bibliotheca Eugeniana",
			"taxonomy": "Wissenklassifikation in der Bibliotheca Eugeniana",
			"taxonomyCircle": "Wissenklassifikation in der Bibliotheca Eugeniana",
		}, "colors": {
			"time": "Zeitlicher Ursprung des Bestands (Klassifikation)",
			"geo": "Räumlicher Ursprung des Bestands (Klassifikation)",
			"language": "Sprachen",
			"geomap": "Landkarte zur Bibliotheca Eugeniana",
			"color": "Klassifikation in der Bibliotheca Eugeniana",
			"taxonomy": "Wissenklassifikation in der Bibliotheca Eugeniana",
			"taxonomyCircle": "Wissenklassifikation in der Bibliotheca Eugeniana",
		},
	}
	
	useEffect(()=>{
		logVersion('eugeniana')
	}, [])
	
	useEffect(() => {
		//console.log(showWelcome)
	}, [showWelcome])
	
	useEffect(() => {
		const handleBeforeUnload = (e) => {
			e.preventDefault();
			e.returnValue = ''; // This shows browser's default dialog
		};
		
		window.addEventListener('beforeunload', handleBeforeUnload);
		
		return () => {
			window.removeEventListener('beforeunload', handleBeforeUnload);
		};
	}, []);
	
	useEffect(() => {
		//console.log(showWelcome)
	}, [showStories])
	
	useEffect(() => {
		useStore.getState().updateLegend();
	}, [chartType])
	
	useEffect(() => {
	}, [mode]);
	
	useEffect(() => {
		setDimchanged(d => !d)
		// console.log('dim changed', screen, left, middle, right)
	}, [screen, left, middle, right, setDimchanged]);
	
	// Utility function to validate and sanitize barcode
	const sanitizeBarcode = (barcode) => {
		if (!barcode) return null;
		
		// Convert to string and uppercase
		const normalizedBarcode = String(barcode).toUpperCase().trim();
		
		// Regular expression to match valid barcode patterns:
		// - Optional plus sign
		// - 1-3 characters
		// - Followed by digits
		const barcodePattern = /^(\+)?[A-Z0-9]{1,3}\d+$/;
		
		if (!barcodePattern.test(normalizedBarcode)) {
			return null;
		}
		
		return normalizedBarcode;
	};
	
	useEffect(() => {
		const { barcode } = queryString.parse(window.location.search);
		
		if (barcode) {
			const sanitizedBarcode = sanitizeBarcode(barcode);
			if (sanitizedBarcode) {
				setIsBarcode(sanitizedBarcode);
				setShowIntro(false);
			} else {
				console.warn(`Invalid barcode format: ${barcode}`);
				// Optionally show an error message to the user
				alert('Ungültiger Barcode. Erwartetes Format: Z1234567 oder ähnliches.');
			}
		}
	}, []);
	
	useEffect(() => {
		if (!store.data) return;
		if (!isBarcode) return;
		
		setSearch(isBarcode);
		setShowIntro(false);
		setShowHighlights(false);
		setChartType('taxonomyCircle');
		
		// Using normalized comparison for case-insensitive matching
		const booksWithSpecificBarcode = store.data.filter(book =>
			typeof book['Barcode'] === 'string' &&
			book['Barcode'].toUpperCase().includes(isBarcode)
		);
		
		const book = booksWithSpecificBarcode[0];
		
		if (!book && isBarcode && search !== '') {
			if (booksWithSpecificBarcode.length < 1) {
				alert(`Leider konnte das Buch mit dem Barcode ${isBarcode} nicht gefunden werden. Bitte überprüfen Sie die URL.`);
			} else if (booksWithSpecificBarcode.length > 1) {
				alert(`Es wurden mehrere Bücher mit dem Barcode ${isBarcode} gefunden. Bitte überprüfen Sie die URL.`);
			}
		}
	}, [store.data, isBarcode]);
	
	const updateDimCallback = useCallback(() => {
		setDim({...dim, screen, top, left, middle, right})
		if (screen.width !== 0 && !firstRender) {
			setFirstRender(true)
		}
	}, [dim, screen, left, middle, right, setDim]);
	
	useEffect(() => {
		updateDimCallback()
	}, [throttledValue]);
	
	useEffect(() => {
		// Gets the root element's font size
		const rootFontSize = parseFloat(getComputedStyle(document.documentElement).fontSize);
		setRemInPx(rootFontSize);
	}, [dim]);
	
	
	const fetchDataCallback = useCallback(() => {
		store.fetchData('data_final.tsv');
	}, [store]);
	
	useEffect(() => {
		fetchDataCallback();
	}, []) // eslint-disable react-hooks/exhaustive-deps
	
	useEffect(() => {
		d3.tsv('BED_geocoded_places.tsv').then((data) => {
			setPlaces(data);
		});
	}, []);
	
	const handleMouse = (e) => {
		if (!showCloseup) setMouseScreen({x: e.clientX, y: e.clientY});
	}
	
	if (showWelcome) {
		return (<ShowWelcome/>)
	}
	
	if (store.loading) {
		return <p>Loading...</p>
	}
	
	if (store.data !== null) {
		return (<div>
			<Screen/>
			<Preprocess/>
			<ShowImpressum/>
			<Stories/>
			<Onboarding/>
			<Tutorial/>
			<Tour />
			
			<div className={styles.container} ref={refScreen}
			     onMouseMove={handleMouse}
			     onTouchStart={() => setWasTouched(true)}
			>
				{!showCloseup && hoveredBook && <TipOverlay/>}
				<ShowHintNonDigitzed/>
				<div className={styles.AppBar} ref={refTop}>
					<NavBar/>
				</div>
				{showHighlights &&
					<div
						style={{position: 'absolute', top: dim.top.height*1.5, right: 0, width: '100%', height: '100%', visibility: !showHighlights? 'hidden': 'visible' }}>
						<Highlights />
					</div>
				}
				<div className={styles.ListView} ref={refLeft}
				     style={{visibility: showHighlights? 'hidden': 'visible'}}
				>
					<div className={styles.ListWrap}>
						<div className={styles.ListPulltab} style={{top: left.height - 15}}></div>
						<BookList height={left.height} width={left.width}/>
					</div>
				</div>
				<div className={styles.OvalView} ref={refMiddle}
				     style={{
					     visibility: showHighlights? 'hidden': 'visible',
				     }}>
					<OvalSvg/>
					{mode === "colors" &&
						<Legend dataset={"colors"}/>
					}
					{mode === "memberships" &&
						<Legend dataset={legend.memberships}/>
					}
					<Hint/>
					<TitleLeft offset={left.width}/>
					{mode && chartType && mode !== "overview" && mode !== "placement" &&
						<TitleRight text={titleChart[mode][chartType]}/>}
				</div>
				<div className={styles.VisView} ref={refRight} onMouseEnter={() => setHoveredBook(false)}
				     style={{
					     visibility: showHighlights? 'hidden': 'visible',
					     overflowX: chartType === "geomap" ? 'visible' : 'hidden',
					     overflowY: chartType === "geomap" ? 'visible' : 'hidden',
					     paddingRight: (chartType === 'member' || chartType === 'color') ? '5%' : '0%',
				     }}
				>
					{mode === 'overview' && <MembershipView/>}
					{mode !== 'overview' && chartType === 'member' && <MembershipView/>}
					{mode !== 'overview' && chartType === 'color' && <ColorView/>}
					{mode !== 'overview' && chartType === 'time' && <TimeLine/>}
					{mode !== 'overview' && chartType === 'geomap' && <GeoMap/>}
					{mode !== 'overview' && chartType === 'taxonomyCircle' && <CirclePacking/>}
					{mode !== 'overview' && chartType === 'geo' && <Places/>}
					{mode !== 'overview' && chartType === 'language' && <Languages/>}
					<div>{showCloseup ? <CombinedViewer identifier={showCloseup}
					                                    offset={0}
					></CombinedViewer> : <></>}</div>
				</div>
				
				<ShowTouch/>
			
			</div>
		</div>);
	}
}

export default App;

const TitleLeft = (props) => {
	const {offset} = props;
	const adjustedOffset = offset + 20; // Adjust offset here if needed
	const style = {left: `${adjustedOffset}px`}; // Convert to string with "px" unit
	
	return (<div className={styles.titleLeft} style={style}>
		Aufstellung im Prunksaal
	</div>);
};

TitleLeft.propTypes = {
	offset: PropTypes.number.isRequired,
};

export const TitleRight = (props) => {
	const {text} = props
	
	return (<div className={styles.titleRight}>
		{text}
	</div>)
}

TitleRight.propTypes = {
	text: PropTypes.string.isRequired,
};


const TipOverlay = React.memo(() => {
	
	const hoveredBook = useStore(state => state.hoveredBook);
	const mouseScreen = useStore(state => state.mouseScreen);
	
	const dim = useStore(state => state.dim);
	
	const [longText, setLongText] = useState()
	
	const ref = useRef();
	
	useEffect(() => {
	}, [hoveredBook]);
	
	useEffect(() => {
		
		if (!hoveredBook) setLongText(' ')
		
		const sep = ", ";
		let text = "";
		
		text += hoveredBook['Signatur_corr']
		text += hoveredBook['Anfang Veröffentlichungsdatum'] && hoveredBook['Anfang Veröffentlichungsdatum'].length > 2 ? sep + hoveredBook['Anfang Veröffentlichungsdatum'] : sep + "[ohne Datum]";
		text += hoveredBook['Orte_corr'] ? sep + hoveredBook['Orte_corr'] : sep + "[ohne Ort]";
		text += hoveredBook['Autor'] ? sep + hoveredBook['Autor'] : sep + "[ohne Autor]";
		text += hoveredBook['Titel'] ? sep + hoveredBook['Titel'] : sep + "[ohne Titel]";
		
		setLongText(text)
		
	}, [hoveredBook]);
	
	if (!mouseScreen) {
		return <></>
	}
	return (<div
			ref={ref}
			className={styles.bookTip}
			style={{
				display: hoveredBook ? 'visible' : 'none',
				top: mouseScreen.y - 70,
				left: mouseScreen.x < dim.left.width ? 0 : mouseScreen.x - 20,
			}}
		
		>
			<LinesEllipsis
				text={longText}
				maxLine='2'
				ellipsis='...'
				trimRight
				basedOn='letters'
			/>
		
		</div>
	
	)
	
})
TipOverlay.displayName = 'TipOverlay';

export const ShowWelcome = () => {
	return (<div className={styles.WelcomeMobile}>
			<div className={styles.WelcomeMobileContent}>
				Der Besuch dieser Bibliotheks-Visualisierung wird Ihnen mit einem größeren Bildschirm viel Freude
				bereiten!
			</div>
			<div className={styles.WelcomeMobileContent}>
				Bitte nutzen Sie idealerweise ein Endgerät ohne Touch-Display ab 1024px Bildschirmbreite.
			</div>
		</div>
	
	)
}

export const ShowTouch = () => {
	
	const wasTouched = useStore(state => state.wasTouched)
	
	const [showTouch, setShowTouch] = useState(true)
	
	return (<>
		{wasTouched && showTouch && <div className={styles.TouchHint}>
			<div className={styles.TouchHintContent}>
				<p>
					Der Besuch dieser Bibliotheks-Visualisierung wird Ihnen <b>ohne Touch-Display</b> viel
					Freude bereiten!
				</p>
				<p>
					Bitte nutzen Sie idealerweise ein Endgerät ohne Touch-Display ab 1024px Bildschirmbreite.
				</p>
				<br/>
				<div className={styles.Button} width={'max-content'} style={{width: 'max-content'}}
				     onClick={() => setShowTouch(false)}>OK
				</div>
			</div>
		</div>}
	</>)
}

export const ShowHintNonDigitzed = () => {
	
	const showHintNonDigitized = useStore(state => state.showHintNonDigitized)
	const setShowHintNonDigitized = useStore(state => state.setShowHintNonDigitized)
	const colors = useStore(state => state.colors)
	
	useEffect(() => {
		// console.log('clicked non digitized')
	}, [showHintNonDigitized]);
	
	return (<>
		{showHintNonDigitized && <div className={styles.HintNonDigitized}>
			<div className={styles.HintNonDigitizedContent}>
				<p>Bitte um Nachsicht, dieses Buch wurde leider noch nicht digitalisiert.
					Wir sind laufend damit beschäftigt, die fehlenden Bücher zu digitalisieren. Bitte versuchen
					Sie es später noch einmal!
				</p>
				
				<div style={{display: 'flex', flexDirection: 'column'}}></div>
				<div className={styles.bookItemIcon}
				     style={{
					     display: 'flex', flexDirection: 'row', flexWrap: 'nowrap', alignItems: 'baseline',
				     }}>
					<BookIcon fill={colors['membership'][0]}
					          strokeWidth={1}
					          innerStroke={colors['membershipInner'][0]}
					/>
					<BookIcon fill={colors['membership'][1]}
					          strokeWidth={1}
					          innerStroke={'white'}
					/>
					<BookIcon fill={colors['membership'][2]}
					          strokeWidth={1}
					          innerStroke={colors['membershipInner'][2]}
					/>
					<div style={{paddingLeft: '1rem'}}>
						Buchsymbole mit Linien stehen für digitalisierte Bücher
					</div>
				</div>
				<div className={styles.bookItemIcon}
				     style={{
					     display: 'flex', flexDirection: 'row', flexWrap: 'nowrap', alignItems: 'baseline'
				     }}>
					<BookIcon fill={colors['membership'][0]}
					          strokeWidth={1}
					          innerStroke={'none'}
					/>
					<BookIcon fill={colors['membership'][1]}
					          strokeWidth={1}
					          innerStroke={'none'}
					/>
					<BookIcon fill={colors['membership'][2]}
					          strokeWidth={1}
					          innerStroke={'none'}
					/>
					<div style={{paddingLeft: '1rem'}}>
						Buchsymbole ohne Linien stehen für Bücher, die noch nicht digitalisiert wurden
					</div>
				</div>
				<br/>
				<div className={styles.Button} width={'max-content'} style={{width: 'max-content'}}
				     onClick={() => setShowHintNonDigitized(false)}>OK
				</div>
			</div>
		</div>}
	</>)
}

export const ShowImpressum = () => {
	
	const showImpressum = useStore(state => state.showImpressum)
	const setShowImpressum = useStore(state => state.setShowImpressum)
	
	return (<div className={styles.imprintContainer} style={{visibility: showImpressum ? 'visible' : 'hidden'}}>
		<div className={styles.imprintContent}>
			<p>
				Diese prototypische Visualisierungs-App entstand im Rahmen des Projektes Bibliotheca Eugeniana Digital.
				Dessen Ziel ist die digitale Rekonstruktion und visuelle Darstellung von Prinz Eugens Büchersammlung
				(UNESCO „Memory of Austria“), einer der berühmtesten Sammlungen der Barockzeit. Die Visualisierung
				basiert auf den Daten einer <a className={styles.imprintLink}
				                               href="https://edition.onb.ac.at/context:eugeniana" target="_blank"
				                               rel="nofollow">digitalen
				Edition des handschriftlichen Bibliothekskatalogs</a>. 
				</p><p>Projektbeteiligte Institutionen:</p>
			<p>
				<a className={styles.imprintLink} href={'https://www.onb.ac.at/'} target="blank" rel="nofollow">
					ONB - Österreichische Nationalbibliothek
				</a>
				<br/>
				<a className={styles.imprintLink} href={'http://www.donau-uni.ac.at/cctc'} target="blank"
				   rel="nofollow">
					UWK - Universität für Weiterbildung Krems</a>
				<br/>
				<a className={styles.imprintLink} href={'http://www.mindfactor.at'} target="blank" rel="nofollow">
					mindfactor IT solutions eU</a>
			</p>
			<p>
				Kontakt: <span className="email"
				               style={{userSelect: 'all'}}>bed-project@onb<b>.onb.</b>.ac.at</span>
			</p>
			<p>
				Das Projekt Bibliotheca Eugeniana Digital wird durch die österreichische Akademie der Wissenschaften
				im Rahmen des Calls <a className={styles.imprintLink}
				                       href={'https://www.oeaw.ac.at/foerderungen/godigital'}
				                       target="blank" rel="nofollow">
				godigital!3.0
			</a> gefördert.
			
			</p>
			<p>
				In der Visualisierungs-App sind Inhalte und Bilder durch externe Links und Viewer
				eingebunden. Die Bild- und Nutzungsrechte dieser Inhalte werden von den Institutionen
				definiert, die diese bereitstellen. Die Bild- und Nutzungsrechte der aktuellen
				Visualisierungs-App liegen bei dem Forschungsprojekt Bibliotheca Eugeniana Digital. Alle im Rahmen des
				Projekts erstellten Daten werden über die <a className={styles.imprintLink}
				                                             href={'https://labs.onb.ac.at/bed/'}
				                                             target="blank"
				                                             rel="nofollow">ÖNB Labs</a> zugänglich gemacht und gemäß FAIR-Prinzipien mit europäischen
				Forschungsinfrastrukturen geteilt.
			</p>
			<p>
				Grundsätzlich werden von der Visualisierungs-App keine Cookies gesetzt und keine persönlichen Informationen
				gespeichert oder an Dritte weitergegeben. Die IP-Adresse wird nur anonymisiert gespeichert, es
				findet kein Tracking mit Google Analytics statt.
			</p>
			<br/>
			
			
			<div className={styles.Button} width={'max-content'}
			     style={{width: 'max-content', marginLeft: '80%', marginRight: 'auto'}}
			     onClick={() => setShowImpressum(false)}>Impressum schließen
			</div>
			
			<div
				className={styles.close}
				onClick={() => setShowImpressum(false)}
			>&#10006;</div>
		
		</div>
	</div>)
	
	
}

export const Tour = () => {
	const showTour = useStore(state => state.showTour);
	const setShowTour = useStore(state => state.setShowTour);
	const dim = useStore(state => state.dim);
	const highlightsOffset = useStore(state => state.highlightsOffset);
	
	useEffect(() => {
		// console.log('clicked tour');
	}, [showTour]);
	
	if (!showTour) return null;
	
	return (
		<div className={styles.tourContainer} style={{visibility: showTour ? 'visible' : 'hidden', width: dim.top.width-highlightsOffset-10, top: dim.top.height+30, left:highlightsOffset+10, height: dim.left.height-30}}>
			<div className={styles.tourContent}>
				<iframe
					src="https://showcase.beko-solutions.at/showcases/Prunksaal-onb-Besucher?applicationKey=dc6b726a183b412d87faca9695804305&ss=9&sr=.33,1.53"
					width="100%"
					height={dim.left.height - 100}
					frameBorder="0"
					allowFullScreen
					title="Prunksaal ONB Besucher Showcase"
				></iframe>
			</div>
			
			<div
				className={styles.Button}
				style={{width: 'max-content', margin: '1em', marginLeft: 'auto', marginRight: '2em'}}
				onClick={() => setShowTour(false)}
			>
				Tour schließen
			</div>
			
			<div
				className={styles.close}
				style={{top: dim.top.height+20, right: 20}}
				onClick={() => setShowTour(false)}
			>
				&#10006;
			</div>
		</div>
	);
};
