# Bibliotheca Eugeniana Digital (BED) Application

v1.0, 2024-11-27

## Introduction

The Bibliotheca Eugeniana Digital (BED) Application is a scientific prototype developed in 2024 as a web
application using ReactJS and Vite. This project aims to digitally visualize the Bibliotheca Eugeniana,
offering an interactive platform for scholarly exploration and research. The BED application provides innovative
ways to interact with and study this historical collection. As a research prototype, this software is designed
for academic purposes and exploratory use; it has not been optimized or hardened for production environments.

In the visualization app, content and images are integrated through external links and viewers. The image
and usage rights for these contents are defined by the institutions that provide them. The image and usage
rights of the current visualization app belong to the research project Bibliotheca Eugeniana Digital. All
data created within the framework of the project will be made accessible via ÖNB Labs and shared with European
research infrastructures in accordance with FAIR principles.

## Software Architecture

### Core Technologies

The application is primarily built using open-source technologies, leveraging the
power and flexibility of ReactJS and Vite for a modern, efficient web development experience.

### Mapping Functionality

For mapping capabilities, the application integrates Mapbox, a paid service that offers
a limited free tier. The architecture supports two modes of operation:

1. **Local Development**: Direct Mapbox API calls for ease of development and testing.
2. **Production Deployment**: A proxy server to manage API requests and implement rate limiting.

It's important to note that both direct and proxied API calls contribute to the Mapbox usage quota.
To ensure responsible usage within the free tier limits, we've implemented the following measures:

- A rate limiter is available for deployment via Docker.
- This limit is reset with each new build deployment.
- The limit counter persists across Docker restarts for accurate tracking.
- The API call limit is set to 50,000 as default, and EVERY call to the mapbox API counts.
- The general limit can be finetuned in server/server.js

   ```
   // Apply global rate limiter to Mapbox API routes
   app.use('/api/mapbox', applyGlobalRateLimit);
   ```

### Mirador Viewer Integration

The application incorporates the Mirador viewer for enhanced digital content interaction.
However, due to compatibility constraints:

- We currently use an outdated version of the Mirador viewer, since no newer version is currently available.
- This necessitates the use of an older node version (18.16.0), which may have potential security implications.

The maintainer of this project should actively explore options to update this component in future
iterations of the project.

## A Brief Word on What You're Getting Into

> *Ladies and gentlemen, gather 'round for a tale of scientific ambition and coding...
let's call it "creativity." This project is what happens when academia meets
software development in a dark alley and neither one backs down. It's a prototype,
which is a fancy way of saying "it works, but don't look too closely at how."*
>
> *Now, you might be thinking, "Surely, the code must be a masterpiece of elegance
and efficiency!" And to that I say: Oh, my sweet summer child. This codebase has
been through more iterations than a politician's stance on a hot-button issue.
It's a beautiful mess, a potpourri of "it seemed like a good idea at the time"
and "dear God, what was I thinking?"*
>
> *But here's the thing: it works. It's not pretty, it's not sleek, and it definitely
won't be featured in "Clean Code Monthly." But like a car held together with duct
tape and optimism, it gets you where you need to go. So buckle up, suspend your
expectations of coding best practices, and enjoy the ride. Who knows? You might
learn something. I certainly did, though I'm still not sure what.*
>

## Prerequisites

- Node.js (v18.16.0 highly recommended, otherwise mirador viewer won't install properly)
- npm (comes with Node.js)
- Docker and Docker Compose (for production deployment)

## Setup

1. Clone the repository:
   ```
   git clone https://github.com/yourusername/your-repo-name.git
   cd your-repo-name
   ```

2. Install dependencies:
   ```
   cd app
   npm install
   ```

3. Install docker and start the daemon (only needed for deployment):

4. Set up environment variables:

   Create a `.env` file in the root directory (your-repo-name):
   Put the token for *deployment* here
   ```
   MAPBOX_TOKEN=your_deployment_mapbox_token

   NODE_ENV=production
   MAPBOX_LIMIT=7500000000
   
   PORT=3010
   
   # Raster tile limits
   LIMIT_RASTER_CACHED=7500000000
   LIMIT_RASTER_UNCACHED=50000
   
   # Vector tile limits
   LIMIT_VECTOR_CACHED=250000000
   LIMIT_VECTOR_UNCACHED=10000
   
   # Font limits
   LIMIT_FONTS_CACHED=50000
   LIMIT_FONTS_UNCACHED=1000
   
   # Style limits
   LIMIT_STYLES_CACHED=50000
   LIMIT_STYLES_UNCACHED=1000
   
   # Sprite limits
   LIMIT_SPRITES_CACHED=50000
   LIMIT_SPRITES_UNCACHED=1000
   
   # Other request limits
   LIMIT_OTHER_CACHED=50000
   LIMIT_OTHER_UNCACHED=1000

   ```

   Create a `.env` file in the app directory:
   Put the production token here.

   ```
      VITE_APP_MAPBOX_ACCESS_TOKEN="your_deployment_mapbox_token"
   ```

   If you have a seperate token for *development*, put this token here instead of the production token.
   ```
      VITE_APP_MAPBOX_ACCESS_TOKEN="your_>>development<<_mapbox_token"
   ```

The setup is complete. Please read the next section about Limiting and caching carefully.
After that section, the last steps to run the app locally for development and also in a docker file for deployment are discussed.

## Limits & Caching

There are different limits for free use when using the app over the dockerfile or without it.
Without the dockerfile, webpage loads are the crucial parameter. When using the dockerfile, the number of 
tile requests will decide about billing or not billing. 

### Development ###
Please do not use the dockerfile for development. Without using the dockerfile, you can 
load the app 50.000 times per month and all tiles are free. For most cases this should be sufficient.
If not, use an additional map-box account for development.

### Deployment ###
Please use the dockerfile for deployment. Tiles will count here and you must
use caching to avoid billing. Do not fill and reset the cache more than 350 times per month.
It is highly recommended that you do not use the dockercontainer when developing, since in development 
the container will be rebuilt each time you change something and then the cache will be reset.

One visit will have a approximate size of 800-2000 raster tiles for the Intro, dependent on screen dimensions. Larger
screens lead to more tiles. The current limit are 750.000 ratsertiles per month.
Also, a handful of vector tiles will be used, dependent on how much the map was used.
Also, a very small amount of other mapbox services is used.

To keep the footprint as small as possible, every mapbox request is cached.
The cache expires after 180 days. Typical cache-size should be around 40-70 MB.

After rebuilding the Docker image, please note that the first few application starts may be slower than usual as the
cache needs time to warm up and reach optimal performance.

#### General Limit

All limits are reset after 30 days. If you rebuild the dockerfile, all limits and the cache are reset.
Use *_UNCACHED limits to avoid billing.
Use *_CACHED limits to avoid a too heavy use of the docker container.

```
MAPBOX-LIMIT=7500000000
```

Every call to mapbox is counted here, no matter if we fetch from cache or fetch from mapbox.
Set this value very high, actuall 7.500.000.000.
This should be enough for approx. 50.000 visits of the app.

#### Service dependent limits

The raster tile limit is the most important one, since many tiles are used in the intro.
Cached raster tiles should be set very high, they should be limited what the dockerfile can process.
Billing starts actually at 7.500.000 tiles, so set unchached tiles below that limit.    
We set uncached limits to 50.000 by default, that should suffice to get all raster tiles.

```
# Vector tile limits
LIMIT_RASTER_CACHED=7000000000
LIMIT_RASTER_UNCACHED=50000
# billing > 750.000

# Vector tile limits
LIMIT_VECTOR_CACHED=250000000
LIMIT_VECTOR_UNCACHED=10000
# billing > 200.000

# Font limits
LIMIT_FONTS_CACHED=50000
LIMIT_FONTS_UNCACHED=1000
# no billing for fonts at the moment

# Style limits
LIMIT_STYLES_CACHED=50000
LIMIT_STYLES_UNCACHED=1000
# billing > 50.000 (maploads)

# Sprite limits
LIMIT_SPRITES_CACHED=50000
LIMIT_SPRITES_UNCACHED=1000
# billing > 50.000 (maploads)

# Other request limits
LIMIT_OTHER_CACHED=50000
LIMIT_OTHER_UNCACHED=1000
# just in case there are other requests
```

### Statistics and Monitoring

The application provides several endpoints to monitor usage and cache performance:

#### Stats Endpoints

1. **Detailed Stats**: `/api/stats/detailed`
   ```json
   {
     "rasterTiles": {
       "cached": 150,    // tiles used from cache
       "fetched": 50,    // tiles fetched from mapbox
       "total": 200
     },
     "vectorTiles": {
       "cached": 300,
       "fetched": 100,
       "total": 400
     },
     // ... similar stats for fonts, styles, sprites, and other
     "totalRequests": 662,
     "uptime": "2d 5h 30m",
     "summary": {
       "totalCached": 500,
       "totalFetched": 162
     }
   }
   ```

2. **Cache Stats**: `/api/cache/stats`
   ```json
   {
     "hits": 500,
     "misses": 162,
     "totalRequests": 662,
     "hitRate": "75.53%",
     "cacheSizeMB": "25.7",
     "filesStored": 550,
     "typeStats": {
       "raster": { "count": 150, "bytes": 15000000, ... },
       "style": { "count": 300, "bytes": 9000000, ... }
     }
   }
   ```

3. **Current Limits**: `/test-mapbox-limit`
   ```json
   {
     "current": 662,
     "resetTime": "2024-12-23T12:00:00.000Z",
     "detailed": {
       // Includes combined stats from both endpoints above
     }
   }
   ```
4. **Service Limits**: `/api/stats/service-limits`
   ```json

   {
   "usage": {
     "rasterTiles": {
       "cached":{
         "count":1793,
         "percentage":"0.00"
        },
      "uncached":{
         "count":1112,
         "percentage":"2.22"
        }
      }
     }
   }
      ```

#### Management Endpoints

1. **Reset Statistics**:
    - Endpoint: `POST /api/stats/reset`
    - Resets all statistical counters while preserving the cache
    - Returns: `{"message": "Stats reset successfully"}`

2. **Reset Cache Stats**:
    - Endpoint: `POST /api/cache/reset`
    - Resets cache statistics while preserving cached files
    - Returns: `{"message": "Cache stats reset successfully"}`

### Usage Monitoring

The stats endpoints can be used to monitor:

- Cache effectiveness (hit rate)
- Resource consumption by type
- Approaching limits
- System uptime
- Cache size and growth

For production monitoring, it's recommended to:

1. Regularly check the detailed stats
2. Monitor the cache size
3. Watch for high uncached request rates
4. Set up alerts when approaching uncached limits
5. Regularily monitor the dashboard of mapbox.com (billing information and stats)

### Cache Management

The cache system:

- Automatically expires entries after 180 days
- Preserves headers for proper content serving
- Maintains cache integrity across restarts
- Optimizes storage using MD5 hashing
- Provides detailed type-specific statistics

The cache can be fully reset by:

1. Rebuilding the Docker container
2. Manually deleting the `tile-cache` directory

## Development

**CAUTION:**: If you run in development described below, you directly call mapbox.com without
using any limiter. Please check that you are in the limits from time to time
in the dashboard of your mapbox studio account. 

On heavy use, you can make a dev account in mapbox and use a dedicated dev
key for VITE_APP_MAPBOX_ACCESS_TOKEN.

To run the application locally with direct calls to Mapbox:

1. Navigate to the `app` directory:
   ```
   cd app
   ```

2. Start the development server:
   ```
   npm run dev
   ```

3. Open your browser and visit `http://localhost:5173` (or the port Vite is running on).

## Production Build

To build the application for production:

1. Navigate to the `app` directory:
   ```
   cd app
   ```

2. Run the build command:
   ```
   npm run build
   ```

   Or, to automatically increment the version build number:
   ```
   npm run build:production
   ```

## Docker Deployment

To deploy the application using Docker:

1. Ensure you're in the project root directory and the docker daemon is running.

2. Build and start the Docker container:
   ```
   docker-compose up --build
   ```

3. The application will be available at `http://localhost:3010` (or the port specified in your Docker configuration).

## Additional Commands

- To run the Docker container:
  ```
  docker-compose up
  ```
- 
- To stop the Docker container:
  ```
  docker-compose down
  ```

- To view Docker logs:
  ```
  docker-compose logs
  ```

## Notes

- The `.env` files contain sensitive information and should not be committed to version control. Make sure they are
  included in your `.gitignore` file.
- In development mode, the application makes direct calls to Mapbox API. In production, it uses a proxy server to manage
  API requests and implement rate limiting.
- Always use `VITE_APP_` prefix for environment variables that need to be accessible in the frontend code.

## Troubleshooting

If you encounter any issues:

- Ensure all environment variables are correctly set.
- Check that you're using the correct Mapbox token for each environment.
- Verify that all required dependencies are installed.
- Check the console and server logs for any error messages.

For any other problems, please open an issue in the project repository.

## Contributors

The concept for the visualizations and UI was developed by Eva Mayr and Florian Windhager (both from UWK),
with feedback loops from the project team and external contributors.

The dataset was extensively reworked by Simon Mayer and Annerose Tartler from ÖNB (Austrian National Library).

Approximately 15% of the code was contributed by Johannes Liem, a former project member.
The remaining code was developed by Michael Smuc (office@mindfactor.at).

For a more comprehensive project description, please visit https://labs.onb.ac.at/bed/
