import requests
import pathlib
import json
import os
from progress.bar import IncrementalBar
from dotenv import load_dotenv

load_dotenv()

def get_content(url):
    retries = 0
    while retries < 3:
        try:
            resp = requests.get(url)
            if resp.status_code == 200:
                return resp.content
            else:
                retries += 1
        except requests.exceptions.Timeout:
            retries += 1
        except requests.exceptions.TooManyRedirects as e:
            raise SystemExit(e)
        except requests.exceptions.RequestException as e:
            raise SystemExit(e)


if __name__ == '__main__':
    api_key = os.getenv('API_KEY')
    img_dir = pathlib.Path('img')
    if not img_dir.exists():
        img_dir.mkdir()
    codices = ['codex_14.376', 'codex_14.377', 'codex_14.378', 'codex_14.379', 'codex_14.380']
    dod_nums = [51202, 51184, 51219, 51191, 51195]
    subdirs = [img_dir.joinpath(codex) for codex in codices]
    for img_subdir, dod_num in zip(subdirs, dod_nums):
        if not img_subdir.exists():
            img_subdir.mkdir()
        cod_url = f'https://astor1.onb.ac.at/access/dip/DOD_{dod_num}?apikey={api_key}'
        manifest = json.loads(get_content(cod_url))
        files = manifest['files']
        img_names_urls = [(files[entry]['filename'].replace('.jp2', '.jpeg'), files[entry]['imageUrl'])
                          for entry in files]
        with IncrementalBar(f'Downloading Codex to {img_subdir}', max=len(img_names_urls)) as bar:
            for name, url in img_names_urls:
                filename = f"{img_subdir.as_posix().replace('img/codex_', '')}_{name}"
                filepath = img_subdir.joinpath(filename)
                if not filepath.exists():
                    img_content = get_content(url)
                    open(filepath, 'wb').write(img_content)
                bar.next()
            bar.finish()
            file_count = len(next(os.walk(img_subdir))[2])
            print(f'Successfully downloaded {file_count} out of {len(img_names_urls)} images')
